/**
 *
 * \file        hprfgw_rfapp.c
 *
 * \brief       Gateway application task
 *
 * \author      Hazrat Shah
 *
 * \date        11/1/2006	
 *
 * \note
 *
 */

////////////////////////////////////////////////////////////////////////////////
#include "hprfgw_config.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfRecMsg.h"
#include "hprfgw_rfDiagnostics.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfSharedDefs.h"
#include "hprfgw_rfApp.h" 
#include "smac_app.h"
#ifdef	RF_NETWORK_MASTER_DEVICE
#include "uart_cresnet.h"
#include "masterlist.h"
#include "network.h"
#include "hprfgw_cresnet.h"
#endif
//#include "../../../../external/crestron/ERInterfaceDLL/nvram_dataMap.h"
#include "inc/hprf_androidIntfc.h"
////////////////////////////////////////////////////////////////////////////////
//	External function declarations
////////////////////////////////////////////////////////////////////////////////
extern INT8 *ConvertUIDtoString (UINT8 *puid, INT8 *pstr, UINT8 addrmode);

////////////////////////////////////////////////////////////////////////////////
#ifndef RF_TASKS_BASE_PRIORITY
 #if defined (OS_NUCLEUS)
  #pragma info "!!!!Pass in the RF_TASKS_BASE_PRIORITY using the make file!!!!"
 #endif
#endif
#define RFAPP_TASK_PRIORITY					RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY, 2)
#define	RFAPP_TASK_PIPE_BUFFER_SIZE			(16*1024)
#define	RFAPP_TASK_STACK_SIZE				(16*1024)
#define	RFAPP_PIPE_MAX_MSG_LEN				(sizeof(TYPE_RFAPP_DATA_BUF))

#define RFAPP_TIMERTASK_PRIORITY			RF_TASKS_BASE_PRIORITY
#define	RFAPP_TIMERTASK_STACK_SIZE			(16*1024)

#if defined (OS_NUCLEUS)
 #define	RFAPP_TIMER_EVENT					0x00000001
 #define	RFAPP_RFPACKET_IN_PIPE_EVENT		0x00000002
 #define	RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT 0x00000004
#elif defined (OS_WINCE)
 // indexes to event array
 #define	RFAPP_TIMER_EVENT					0x00000000
 #define	RFAPP_RFPACKET_IN_PIPE_EVENT		0x00000001
 #define	RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT 0x00000002

 #define	RFAPP_EVENT_COUNT					3			// total events
#endif

#define	RFAPP_TIMER_EVENT					(1 << 0)
#define	RFAPP_RFPACKET_IN_PIPE_EVENT		(1 << 1)
#define	RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT (1 << 2)
#define RFAPP_EVENT_MASK   (RFAPP_TIMER_EVENT | RFAPP_RFPACKET_IN_PIPE_EVENT | RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT)
#define	RFAPP_EVENT_COUNT					3			// total events


#if defined (OS_NUCLEUS)
 #define	RFAPP_TASK_TIMER_RATE				(1*TICKS_PER_SECOND)							//	1-second
#elif defined (OS_WINCE)
 #define	RFAPP_TASK_TIMER_RATE				(1000)											//	1-second
#endif
#define     RFAPP_TASK_TIMER_RATE				(1000)											//	1-second
#define	RFAPP_CRESNET_CONNECTION_TIMEOUT	((6*RFAPP_TASK_TIMER_RATE)/RFAPP_TASK_TIMER_RATE)	//	6-seconds

////////////////////////////////////////////////////////////////////////////////
//	RF network/application version number
#define RF_VERSION			  "1.01.04"
#define RF_VERCOMPLETE		  (" Ver " RF_VERSION " Build: " __DATE__ " " __TIME__)
const unsigned char RF_Version[] =  RF_VERCOMPLETE;

typedef struct	{
#if defined (OS_NUCLEUS)
	NU_EVENT_GROUP events;
	NU_TIMER	   rappTimer;

	NU_PIPE	 pipe;						
	UINT8	 pipebuf[RFAPP_TASK_PIPE_BUFFER_SIZE];					

	NU_TASK  task;
    UINT8 stack[RFAPP_TASK_STACK_SIZE];
#elif defined (OS_WINCE)
	HANDLE	 hEvents[RFAPP_EVENT_COUNT];
	HANDLE	 rappTimerThread;
	HANDLE   rappTimerEvent;
	
	HANDLE	 hWritePipe;
	HANDLE	 hReadPipe;

	HANDLE   task;
#endif
//	DM_RESOURCE_HANDLE	 hEvents[RFAPP_EVENT_COUNT];
	DM_RESOURCE_HANDLE   event_group;
	DM_RESOURCE_HANDLE	 rappTimerThread;
	DM_RESOURCE_HANDLE   rappTimerEvent;

	DM_RESOURCE_HANDLE	 hPipe;

	DM_RESOURCE_HANDLE   task;
} TYPE_RFAPP_TASK;
TYPE_RFAPP_TASK RFAppTask;

//RF roaming task
TYPE_RFAPP_TASK RfRmTaskStr;

typedef struct	{
	BOOL	ControlSystemConn;
	BOOL	CresnetConnected;
	BOOL	CIPOnLine;

	UINT8	NetMonState;
	UINT32	NetMonTimer;
	
	BOOL	cresnetpolling;
	UINT32	cresnetconnectiontimer;
}	TYPE_RFAPP_NET_CONNMON;
TYPE_RFAPP_NET_CONNMON RFAppNetMon;
// ctlsys connection states
#define	RFAPP_CIP_ONLINE		0
#define	RFAPP_CIP_OFFLINE		1
#define	RFAPP_CRESNET_CONN		2
#define	RFAPP_CRESNET_DISCONN	3

//	device info data fields 
TYPE_RF_DEVICE_INFO_REC RFAPPDevDataField;

// Link request/reply handler parameters 
#define	RFLINKREQ_MAX_DELAY_CNT	30
const UINT8	RFLinkReqDelays[RFLINKREQ_MAX_DELAY_CNT] = {
															0,0,0,0,0,0,0,0,0,0,
															1,1,1,1,1,1,1,1,1,1,
															2,2,2,2,2,2,3,3,4,10
														};	//	to avoid draining the battery the link request
															//	rate is reduced to 10-sec after max-retries															
typedef	struct	{
	UINT8	state;
	UINT32	attempts;
	UINT32	timer;
	UINT16	randomno;
	UINT32	retryhighlevel;
	UINT32	retrylowlevel;
	UINT32	prequeryattempts;
}	TYPE_RF_LINKREQ_REPLY_REC;
TYPE_RF_LINKREQ_REPLY_REC RFLink;

//	Acquire process parameters 
typedef	struct	{
	UINT8	state;				//	acquire process current state
	UINT8	acquireattempts;	//	acquire process attempts (incremeted on channel 11...26 loop)
	UINT8	channel;			//	rf channel in acquire process 
	UINT32	timer;				//	acquire process state timer
	UINT16	randomno;			//	random no used in link request during acquire
	UINT32	processtimer;		//	acquire process timeout (when timer reaches 0 acquire is stopped)
	BOOL	acquiresuccessful;		//	device was just acquired successfully
	TYPE_GATEWAY_IN_ACQUIRE_LIST GWAcq[MAX_GATEWAY_IN_ACQUIRE_REC_SIZE];	//	gateway list that are in acquire 
	UINT8	noofGWinAcquire;	//	total number of gatewa's in acquire mode
}	TYPE_RF_ACQUIRE_REC;
TYPE_RF_ACQUIRE_REC	RFAcquire; 

// Echo request/reply handler parameters 
#define	RFAPP_ECHO_REPLY_PACKET_TIMEOUT	5
typedef	struct	{
	TYPE_RF_ECHO_REQUEST_PACKET EchoReq;
	TYPE_RF_ECHO_REPLY_PACKET   EchoReply;	
	UINT32	starttick;
	UINT32	starttxpktcnt;
	UINT32	starttxdatabytecnt;
	UINT32	startrxpktcnt;
	UINT32	startrxdatabytecnt;
	UINT32	timer;
}	TYPE_RF_ECHO_PKT_REC;
TYPE_RF_ECHO_PKT_REC RFEcho;

// Generic request/reply parameters 
#define	RFGEN_REQ_REPLY_MAX_ENTRIES	10		
typedef	struct	{
	UINT16	requesternode;
	UINT16	requesterid;
	UINT16	requesterpar1;
	UINT16	requesterpar2;
	
	UINT16	replynode;
	UINT16	replyid;
	UINT16	replypar1;
	UINT16	replypar2;

	void	(*pfunc) (UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 BufLen);	
}	TYPE_RFGENERIC_REQ_REPLY_REC;
TYPE_RFGENERIC_REQ_REPLY_REC RFGenReqReply[RFGEN_REQ_REPLY_MAX_ENTRIES];


//	RF discover handler states
enum	{	
	RFDISCOVER_IDLE_STATE,			
	RFDISCOVER_INIT_STATE,
	RFDISCOVER_REQUEST_STATE,			
	RFDISCOVER_REPLY_STATE		
};
#define	RFDISCOVER_REPLY_DELAY			2		//	2-seconds wait time per-channel 
#define RFDISCOVER_MAX_REPEATS			0		
//	RF network discover parameters
#define	MAX_AUTO_DISCOVERED_DEVICES		100
typedef	struct	{
	TYPE_RF_DISCOVER_REQUEST req;
	TYPE_RF_DISCOVER_REPLY	 reply[MAX_AUTO_DISCOVERED_DEVICES];

	UINT32	cnt;
	UINT32	state;
	UINT32	timer;	
	UINT32	scannedchannel;
	UINT32	inchrepeat;
}	TYPE_RF_DISCOVER_REC;
TYPE_RF_DISCOVER_REC RFDiscover;

UINT32	NetworkServiceEnableFlags;	
#ifndef	RF_NETWORK_MASTER_DEVICE
enum	{	
	POWER_UP_SKIP = 0,
	POWER_UP_SEND_REQ ,				
	POWER_UP_CHECK_ROAMING_TABLE
};
INT8    powerUpScan = POWER_UP_SEND_REQ;
#endif
//	Trigger sources 
#define		RFREQPKT_TRIGSRC_TIMER_EXPIRED	0
#define		RFREQPKT_TRIGGER_REPLY_PACKET	1	

//	RFLink handler function parameters 
//	RFLink handler function states
enum	{	RFLINKREQ_IDLE_STATE=0,				RFLINKREQ_REQREPLY_INIT_STATE,
			RFLINKREQ_REQUEST_CHKLQI_STATE,		RFLINKREQ_REQUEST_LQIREPLYWAIT_STATE, 
			RFLINKREQ_REQUEST_STATE,			RFLINKREQ_REPLYWAIT_STATE	
};

//	RFAcquire handler function parameters
#define	RFACQUIRE_CHANNEL_SWITCH_DELAY		1	//	1-second	
#define	RFACQUIRE_MAX_ACQUIRE_ATTEMPTS		2	//		
#define	RFACQUIRE_LINKREQ_REPLY_DELAY		1	//	1-seconds
#define	RFACQUIRE_LINKREQ_REPLY_MAX_DELAY	5	//	5-second - max acquire delay 
enum	{
    RFACQUIRE_IDLE_STATE=0,
    RFACQUIRE_INIT_STATE,
    RFACQUIRE_REPEAT_STATE,
    RFACQUIRE_SWITCH_CHANNEL_STATE,
    RFACQUIRE_LINKREQ_STATE,
    RFACQUIRE_LINKREQ_REPLY_STATE
};

//	Network statistics
typedef	struct	{
	UINT32	TxTurnArounddlyCong;
	UINT32	NumTxPktsCong;	
	UINT32	NumRxPktsCong;	
	
	UINT32	NumrxPktsUtil;
	UINT32	NumtxPktsUtil;
}	TYPE_NET_STAT_PARS;
TYPE_NET_STAT_PARS NetBwSt;
UINT32 Rfdiags_Congestion;
UINT32 Rfdiags_Utilization;

//for roaming
#define RFRM_READY 0

#define GW_RM_THRESHOLD_MAX_LEVEL 5
#define GW_RM_THRESHOLD_5DB 10
#define GW_RM_THRESHOLD_10DB 20
#define GW_RM_THRESHOLD_15DB 30
#define GW_RM_THRESHOLD_20DB 40
#define GW_RM_THRESHOLD_25DB 50
#define GW_RM_THRESHOLD_30DB 60
#define GW_RM_THRESHOLD_35DB 70
#define GW_RM_THRESHOLD_40DB 80
#define GW_RM_THRESHOLD_50DB 100
#define GW_RM_THRESHOLD_60DB 120
#define GW_RM_THRESHOLD_65DB 130
#define GW_RM_THRESHOLD_70DB 140
#define GW_RM_THRESHOLD_75DB 150
#define GW_RM_THRESHOLD_80DB 160
#define GW_RM_THRESHOLD_85DB 170
#define GW_RM_THRESHOLD_90DB 180
#define GW_RM_THRESHOLD_92DB 184
#define GW_RM_THRESHOLD_95DB 190
const UINT8 GW_Aggressiveness[GW_RM_THRESHOLD_MAX_LEVEL] =
{
    GW_RM_THRESHOLD_10DB,
    GW_RM_THRESHOLD_15DB,
    GW_RM_THRESHOLD_20DB,
    GW_RM_THRESHOLD_25DB,
    GW_RM_THRESHOLD_30DB
};
#define GATEWAY_OVERFLOW_THRE       10
UINT16  gGW_RM_threshold = GW_RM_THRESHOLD_20DB;//GW_Aggressiveness[ROAMING_AGG_DEFAULT];
bool    g_roamingInProgress = false;
UINT32  roamingStartTime = 0;
INT8    g_GW_switch = 0;
INT16   g_testRoamingBufSize = GW_RM_DEFALT_MV_AV_FILTER_SZ;
////////////////////////////////////////////////////////////////////////////////
//	Local function declarations
////////////////////////////////////////////////////////////////////////////////
void RFAPP_AcquireHandler(UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len);
void RFAPP_AcquireHandlerTimer(void);
void RFAPP_EchoRequestHandler(UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len);
void RFAPP_EchoRequestHandlerTimer (void);	
void RFAPP_DeviceDataInfoInit (void);
void RFAPP_ProcessDeviceDataInfoStrRequest (UINT8 index, UINT8 rfid, TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY *pd, UINT16 len);
void RFAPP_ProcessDeviceDataInfoStrReply (UINT8 index, UINT8 rfid, TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY *pd, UINT16 len);

void RFAPP_LinkRequestHandler(UINT32 trigger, UINT32 par2, UINT8 *pBuf, UINT16 len);
void RFAPP_LinkRequestHandlerTimer(void);	
void RFAPP_LinkRequestHandlerStart(UINT8 delay);	
void RFAPP_LinkRequestHandlerStop (void);
void RFAPP_ProcessLinkReply (UINT8 *pdata, UINT16 len);
void RFAPP_RFNetworkDiscoverHandlerTimer (void);
#ifndef	RF_NETWORK_MASTER_DEVICE
void RFAPP_ScanGatewayHandler(void);
#endif
INT32 RFAPP_RFNetworkDiscoverHandler (UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len);
void  RFAPP_NetworkStatistics (void);
void RFAPP_ControlSystemStatusHandler (UINT32 par, UINT32 newstate);
void RFAPP_ControlSystemStatusHandlerTimer (void);

UINT32 Rfdiags_getCongestion(void) { return Rfdiags_Congestion; }
UINT32 Rfdiags_getUtilization(void) { return Rfdiags_Utilization; }

//RF roaming
void RFRM_Init(void);
static void RFRM_Task(UINT32 param);
void RFRM_updateRoamingGW_connectedIndex(void);
INT8 RF_RM_findCurrentConnectedByIndex(INT8* index);
void RF_RM_SwitchGWByIndex(INT8 index);
////////////////////////////////////////////////////////////////////////////////

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       RF application task timer 
 * 
 * \detail      
 *
 * \return      UNSIGNED ignore
 *
 * \retval      None
 *
 */
#if defined (OS_NUCLEUS)
void RFAPP_Timer (UNSIGNED ignore)	{

	TYPE_RFAPP_TASK* pTask = &RFAppTask;
	NU_Set_Events(&pTask->events, RFAPP_TIMER_EVENT,  NU_OR);
}
#elif defined (OS_WINCE)
DWORD RFAPP_Timer (void *pContext)
{
	TYPE_RFAPP_TASK* pTask = &RFAppTask;

	// Log message to know the task started
	RFIntHostApp.pF_ErrorPrintf("\r\nRFAPP_Timer thread: Running!");
	
	while (1)
	{
		//Sleep (RFAPP_TASK_TIMER_RATE);
		if (WAIT_TIMEOUT == WaitForSingleObject(pTask->rappTimerEvent, RFAPP_TASK_TIMER_RATE))
		{
			SetEvent(pTask->hEvents[RFAPP_TIMER_EVENT]);
		}
		else
		{
			RFIntHostApp.pF_ErrorPrintf("RFAPP_Timer Timeout fail: %d", GetLastError());
		}
	}
}
#endif
void RFAPP_Timer(void) {
//	TYPE_RFAPP_TASK *pTask = &RFAppTask;
	OsSetEvent(RFAppTask.event_group, RFAPP_TIMER_EVENT);
}

UINT16	RFAPP_FindEmptyEntryRFPendingPacketReq (UINT8 *index)	{	
	UINT32	i;
	UINT16  res = FAILURE;

	for (i=0; i<RFGEN_REQ_REPLY_MAX_ENTRIES; i++)	{
		if (!RFGenReqReply[i].pfunc)	{	
			*index = (UINT8)i;
			res = SUCCESS;
			break;
		}
	}
	
	return (res);
}

UINT16	RFAPP_FindEntryofGivenRFPendingPacketReq   (void *pFunc, UINT16 requesterid, UINT16 requesternode, UINT8 *index)	{	
	UINT32	i;
	UINT16  res = FAILURE;

	for (i=0; i<RFGEN_REQ_REPLY_MAX_ENTRIES; i++)	{
		if ((RFGenReqReply[i].pfunc == pFunc) && (RFGenReqReply[i].requesterid==requesterid) && (RFGenReqReply[i].requesternode==requesternode) && (RFGenReqReply[i].pfunc))	{	
			*index = (UINT8)i;
			res = SUCCESS;
			break;
		}
	}
	
	return (res);
}

UINT16	RFAPP_FindEntryofGivenRFPendingPacketReqByID (UINT16 requesterid, UINT16 requesternode, UINT16 replyid, UINT8 *index)	{	
	UINT32	i;
	UINT16  res = FAILURE;

	for (i=0; i<RFGEN_REQ_REPLY_MAX_ENTRIES; i++)	{
		if ((RFGenReqReply[i].requesterid==requesterid) && (RFGenReqReply[i].requesternode==requesternode) && (RFGenReqReply[i].replyid==replyid))	{	
			*index = (UINT8)i;
			res = SUCCESS;
			break;
		}
	}
	
	return (res);
}

UINT16	RFAPP_AddEntryofGivenRFPendingPacketReq (void *pfunc, UINT16 requesterid, UINT16 requesternode, UINT16 replyid)	{	
	//UINT32	i;
	UINT16  res = SUCCESS;
	UINT8	index;

	if			(SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReq(pfunc, requesterid, requesternode, &index))	{
		//	Existing entry, overwrite current entry	
	}	else if (SUCCESS == RFAPP_FindEmptyEntryRFPendingPacketReq  (&index))	{
		//	An empty entry was found	
	}	else	{
		//	All entries are full
		res = FAILURE;
	}

	if (res == SUCCESS)	{
		RFGenReqReply[index].pfunc			=	pfunc;
		RFGenReqReply[index].requesterid	=	requesterid;
		RFGenReqReply[index].requesternode	=	requesternode;
		RFGenReqReply[index].replyid		=	replyid;
	}
	
	return (res);
}

UINT16	RFAPP_DeleteEntryofGivenRFPendingPacketReq (void *pfunc, UINT16 requesterid, UINT16 requesternode)	{	
	UINT16	res = FAILURE;
	UINT8	index;

	if	(SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReq(pfunc, requesterid, requesternode,  &index))	{
		memset ((UINT8*) &RFGenReqReply[index], 0, sizeof(RFGenReqReply[0]));
		res = SUCCESS;
	}

	return (res);
}


/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Process RF application packets eg link request, acquire   
 * 
 * \detail      
 *
 * \return      TYPE_RFAPP_DATA_BUF *prfapp, UINT16 len
 *
 * \retval      UINT16
 *
 */
UINT16	RFAPP_Process_RFPackets (TYPE_RFAPP_DATA_BUF *prfapp, UINT16 len)	{
	UINT8	*pdata = (UINT8*) (&prfapp->data);
	UINT16	res    = SUCCESS;
	UINT8	index;
	UINT16	datalen = prfapp->pkthdr.length+sizeof(prfapp->pkthdr.length)-sizeof(TYPE_INTERNAL_RF_PACKET_HEADER);
	panDescriptor_t *pPanDesc = (panDescriptor_t*) (pdata+1);
	UINT32	i;
	BOOL	isTraceEnabled;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	switch (prfapp->pkthdr.type)	{
		case	RF_PACKET_ECHO_REPLY:								
			if (SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReqByID (RF_PACKET_ECHO_REQUEST, 0, RF_PACKET_ECHO_REPLY, &index))	{
				(RFGenReqReply[index].pfunc) (RFREQPKT_TRIGGER_REPLY_PACKET, 0, pdata, datalen);			
			}	else	{
				//	No one is waiting for this packet
			}
			break;

		case 	RF_PACKET_LINK_REPLY:		
		#ifndef	RF_NETWORK_MASTER_DEVICE
			if (SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReqByID (RF_PACKET_LINK_REQUEST, 0, RF_PACKET_LINK_REPLY, &index))	{
				(RFGenReqReply[index].pfunc) (RFREQPKT_TRIGGER_REPLY_PACKET, 0, pdata, datalen);			
			}	else	{			
				//	No one is waiting for this packet, 
				//	the gateway sends this packet to the device when device loses connection and device internal state 
				//	indicates that has connection  
				RFAPP_ProcessLinkReply (pdata, datalen);
			}
		#endif			
			break;

		case	RF_PACKET_ACTIVE_SCAN_REPLY:
			if (isTraceEnabled)	{					
				//	Check if the scan resulted in any coordinator responses.	
				if(*pdata != 0)	{
					for(i=0; i<*pdata; i++, pPanDesc++)	{
						RFIntHostApp.pF_ConsolePrintf (" ----- HPRFGW %d ----- \r\n", i);
						RFIntHostApp.pF_ConsolePrintf ("Address %02x%02x%02x%02x%02x%02x%02x%02x\r\n",pPanDesc->coordAddress[0],pPanDesc->coordAddress[1],
																					  pPanDesc->coordAddress[2],pPanDesc->coordAddress[3],
																					  pPanDesc->coordAddress[4],pPanDesc->coordAddress[5],
																					  pPanDesc->coordAddress[6],pPanDesc->coordAddress[7]);
						RFIntHostApp.pF_ConsolePrintf ("Address mode %x \r\n",		pPanDesc->coordAddrMode);
						RFIntHostApp.pF_ConsolePrintf ("PAN ID %02x%02x \r\n",		pPanDesc->coordPanId[0],pPanDesc->coordPanId[1]);
						RFIntHostApp.pF_ConsolePrintf ("Channel %02x \r\n",			pPanDesc->logicalChannel);
						RFIntHostApp.pF_ConsolePrintf ("Beacon Spec %02x%02x \r\n", pPanDesc->superFrameSpec[0], pPanDesc->superFrameSpec[0]);
						RFIntHostApp.pF_ConsolePrintf ("Link Quality %02x \r\n",	pPanDesc->linkQuality);
					}
				}	else	{
					RFIntHostApp.pF_ConsolePrintf ("No active network members were found \r\n");
				}
			}

			if (SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReqByID (RF_PACKET_ACTIVE_SCAN_REQUEST, 0, RF_PACKET_ACTIVE_SCAN_REPLY, &index))	{
				(RFGenReqReply[index].pfunc) (RFREQPKT_TRIGGER_REPLY_PACKET, 0, pdata, datalen);			
			}	else	{
				//	No one is waiting for this packet
			}
			break;

		case	RF_PACKET_ENERGY_SCAN_REPLY:
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("\r\n");
				for(i=0; i<(gLogicalChannel26_c-gLogicalChannel11_c+1); i++)	 {
					RFIntHostApp.pF_ConsolePrintf (" Channel %d Energy level %d\r\n", i+gLogicalChannel11_c, *(pdata+i));
				}
			}
					
			if (SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReqByID (RF_PACKET_ENERGY_SCAN_REQUEST, 0, RF_PACKET_ENERGY_SCAN_REPLY, &index))	{
				(RFGenReqReply[index].pfunc) (RFREQPKT_TRIGGER_REPLY_PACKET, 0, pdata, datalen);			
			}	else	{
				//	No one is waiting for this packet
			}
			break;

		case	RF_PACKET_DEVICE_DATA_FIELD_REQUEST:
			//	process device info str request packet
			RFAPP_ProcessDeviceDataInfoStrRequest(prfapp->inthdr.index, prfapp->inthdr.rfid, (TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY*)pdata, datalen);
			break;

		case	RF_PACKET_DEVICE_DATA_FIELD_REPLY:
			//	process device info str reply packet
			RFAPP_ProcessDeviceDataInfoStrReply(prfapp->inthdr.index, prfapp->inthdr.rfid, (TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY*)pdata, datalen);
			break;

		case	RF_PACKET_NETWORK_SERVICES:			
			if (((TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)(pdata))->msgtype == RF_PACKET_DISCOVER_NETDEV_REPLY)	{
				if (SUCCESS == RFAPP_FindEntryofGivenRFPendingPacketReqByID (RF_PACKET_DISCOVER_NETDEV_REQUEST, 0, RF_PACKET_DISCOVER_NETDEV_REPLY, &index))	{
					(RFGenReqReply[index].pfunc) (RFREQPKT_TRIGGER_REPLY_PACKET, 0, (UINT8*)prfapp, len);	
				}	else	{
					//	No one is waiting for this packet
				}			
			}
			break;

		default:
			res = FAILURE;	
			break;
	}		

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called by the Cresnet HISR whenever the Cresnet is polled    	 	   	     			           
 *
 * \detail		
 *
 *						
 * \date        04/03/2007
 *					
 * \param	    None
 *
 * \return		None 					
 *
 * \retval		None	
 */
void RFAPP_SerialCresnetConnectionMonitor (void)	{		
#ifdef	RF_NETWORK_MASTER_DEVICE
	if (UartIsCresnetPolled())	{	
		RFAppNetMon.cresnetconnectiontimer = 0;
		if (!RFAppNetMon.cresnetpolling)	{
			RFAppNetMon.cresnetpolling = TRUE;				
			RFAPP_ControlSystemStatusHandler (0, RFAPP_CRESNET_CONN);		
		}
	}	else	{
		if (RFAppNetMon.cresnetpolling){
			if (++RFAppNetMon.cresnetconnectiontimer > RFAPP_CRESNET_CONNECTION_TIMEOUT)	{
				RFAppNetMon.cresnetconnectiontimer = 0;
				RFAppNetMon.cresnetpolling = FALSE;
				if (CresnetIsEnabled())	{
					RFAPP_ControlSystemStatusHandler (0, RFAPP_CRESNET_DISCONN);
				}
			}
		}
	}
#endif
}

/**
 * \author      Hazrat Shah	
 *
 * \date        11/01/2006
 *
 * \return      
 *
 * \retval      void
 *
 * \brief       Gateway application task
 *
 * \param       argc
 * \param       argv
 *
 * \note
 *
 * \warning
 *
 */
#if defined (OS_NUCLEUS)
static void RFAPP_Task(UINT32 argc, void * argv)
#elif defined (OS_WINCE)
static DWORD RFAPP_Task(void *pContext)
#endif
static void RFAPP_Task(UINT32 param)
{
	UINT8	buf[RFAPP_PIPE_MAX_MSG_LEN];
	UINT32	len;
	TYPE_RFAPP_TASK* pTask = &RFAppTask;
	UINT32	events;
	const TYPE_INTERFACE_TO_RFAPP_APP *pRFA = &RFIntRFApp;
#if defined (OS_WINCE)
	DWORD  dwResult, dwFlags;
#endif

	RFIntHostApp.pF_StartupWait(HPRF_STARTUP_APP_READY);
#if defined (OS_NUCLEUS)
	NU_Control_Timer (&pTask->rappTimer, NU_ENABLE_TIMER);
#elif defined (OS_WINCE)
	//SetTimer(NULL, 0, RFAPP_TASK_TIMER_RATE, (TIMERPROC)RFAPP_Timer );
	if (0xFFFFFFFF == ResumeThread(pTask->rappTimerThread))
	{
		RFIntHostApp.pF_ErrorPrintf("RFAPP_Task: RFApp timer start fail");
	}
#endif
    pTask->rappTimerThread = OsStartTimerPri(RFAPP_TASK_TIMER_RATE, RFAPP_Timer, true, RF_TIMER_THREAD_PRIORITY);
    if (!pTask->rappTimerThread)
    {
        RFIntHostApp.pF_ErrorPrintf("RFAPP: Fail create Timer Thread!");
    }

	//	Wait for RF Stack to be intialized
	while (!RFReceive_IsRFStackInitialized())	{	RFIntHostApp.pF_HwDelayMsec(100);	}
	
#ifndef	RF_NETWORK_MASTER_DEVICE
	pRFA->pF_LookupSettingsAndConnectDeviceToNetwork();
#endif

#if defined (OS_WINCE)
	RFIntHostApp.pF_ErrorPrintf("\r\nRFAPP_Task: Running!");
#endif

    while(1)	{				
#if defined (OS_NUCLEUS)
		if (NU_Retrieve_Events(&pTask->events, 0xffffffff, NU_OR_CONSUME, &event_group, NU_SUSPEND) == NU_SUCCESS)	{
#elif defined (OS_WINCE)
		dwResult = WaitForMultipleObjects(RFAPP_EVENT_COUNT, pTask->hEvents, FALSE, OS_SUSPEND);
		if (dwResult >= WAIT_OBJECT_0 && dwResult < RFAPP_EVENT_COUNT)
		{
			//Get Index
			dwResult = dwResult - WAIT_OBJECT_0;
			event_group = dwResult;
#endif
		events = OsWaitForEventAndClear(pTask->event_group, RFAPP_EVENT_MASK);
		if(events) {
#if defined (OS_NUCLEUS)
			if (event_group & RFAPP_TIMER_EVENT)	{
#elif defined (OS_WINCE)
			if (event_group == RFAPP_TIMER_EVENT)	{
#endif
			if(events & RFAPP_TIMER_EVENT) {
				//	Preform background service
				RFAPP_AcquireHandlerTimer ();
            #ifndef	RF_NETWORK_MASTER_DEVICE
				RFAPP_ScanGatewayHandler();       //to scan active gateways at power up
            #endif                
				RFAPP_LinkRequestHandlerTimer ();
				RFAPP_EchoRequestHandlerTimer ();
				RFAPP_RFNetworkDiscoverHandlerTimer ();
				RFAPP_NetworkStatistics();

				//	Monitor Cresnet connection 
				RFAPP_SerialCresnetConnectionMonitor(); 
				RFAPP_ControlSystemStatusHandlerTimer ();
			}	

#if defined (OS_NUCLEUS)
			if (event_group & RFAPP_RFPACKET_IN_PIPE_EVENT)	{
				while (NU_Receive_From_Pipe(&pTask->pipe, buf, RFAPP_PIPE_MAX_MSG_LEN, &len, NU_NO_SUSPEND) == NU_SUCCESS)	{
#elif defined (OS_WINCE)
			if (event_group == RFAPP_RFPACKET_IN_PIPE_EVENT)	{
				while(TRUE == ReadMsgQueue(pTask->hReadPipe, (void*)buf, sizeof (buf), &len, OS_NO_SUSPEND, &dwFlags)) {
#endif
			if(events & RFAPP_RFPACKET_IN_PIPE_EVENT) {
			    // Parens quiet compiler about using assignment as truth value.
			    while(1) {
			        len = OsDequeueBytesWait(pTask->hPipe, (void*)buf, sizeof(buf), OS_NO_WAIT);
			        HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i <--- dequeued %li bytes from hPipe\n", \
		                    __FUNCTION__, __LINE__, len);)
			        if (len <= 0) {
			            break;
			        }
                    //  Process RF received packet
			        RFAPP_Process_RFPackets ((TYPE_RFAPP_DATA_BUF*) buf, len);
			    }
			}

#ifdef	RF_NETWORK_MASTER_DEVICE
#if defined (OS_NUCLEUS)
			if (event_group & RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT)	{
#elif defined (OS_WINCE)
			if (event_group == RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT)	{
#endif
			if(events & RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT) {
				RFIntHostApp.pF_ErrorHandler("ERROR: Cresnet Buffer overflow occured\r\n");					
				if (!IsCIPLinkEstablishedwithCS())	{
					//	broadcast cresnet buffer overflow message to all slave devices
					RFTransmit_BuildBufferOverflowPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST, FALSE, TRUE);
				} 
				//	Report all extender latest values
				ReportAllCresentExtenderAnalogValues();
			} 
#endif
		}
		else
		{
			RFIntHostApp.pF_ErrorPrintf("RFAPP_Task: Fail Event");
		}
    } 
}


/**
 * \author      Hazrat Shah	
 *
 * \date        11/01/2005
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief
 *
 * \param       void
 *
 * \note
 *
 * \warning
 *
 */
void RFAPPInit(void)
{
    TYPE_RFAPP_TASK* pTask = &RFAppTask;
#if defined(OS_NUCLEUS) || defined (OS_WINCE)
    UINT32 inst = 0;
#endif
	BOOL	uidvalid;
#if defined (OS_WINCE)
	MSGQUEUEOPTIONS sMsgQueueOpt;
#endif

	RFIntHostApp.pF_IsUIDValid(&uidvalid);
	
	if (uidvalid)	{
		//	Create event group
#if defined (OS_NUCLEUS)
		if	(NU_Create_Event_Group(&pTask->events, "RFAPPEV") != NU_SUCCESS)	{
			RFIntHostApp.pF_ErrorHandler("RFAPP: Could not create rf application task event group!");
		}	

		//	Create a pipe for messages
		if (NU_Create_Pipe(&pTask->pipe, "PRFAPP", pTask->pipebuf, sizeof(pTask->pipebuf), NU_VARIABLE_SIZE, RFAPP_PIPE_MAX_MSG_LEN, NU_FIFO) != NU_SUCCESS)	{
	   			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);    		   
		}

		memset((UINT8*) &RFGenReqReply, 0, sizeof(RFGenReqReply));

		//	Create a 1-sec generic timer
		if (NU_SUCCESS != NU_Create_Timer (&pTask->rappTimer, "RFAPPTmr", RFAPP_Timer, 0, 1, RFAPP_TASK_TIMER_RATE, NU_DISABLE_TIMER))	{
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
 		}	

		//	Create rf application task
		if(NU_Create_Task(&pTask->task, "TRFAPP", RFAPP_Task, inst, 0, pTask->stack, sizeof(pTask->stack), 
			RFAPP_TASK_PRIORITY, 0, NU_PREEMPT, NU_START) != NU_SUCCESS)	{
			RFIntHostApp.pF_ErrorHandler("RFAPPTASK: Could not create task!");
		}
#elif defined (OS_WINCE)
		//if	(NU_Create_Event_Group(&pTask->events, "RFAPPEV") != NU_SUCCESS)	{
		//	RFIntHostApp.pF_ErrorHandler("RFAPP: Could not create rf application task event group!");
		//}

		for (inst = 0; inst < RFAPP_EVENT_COUNT; inst++)
		{
			pTask->hEvents[inst] = CreateEvent(NULL, FALSE, FALSE, NULL);
			if (pTask->hEvents[inst] == NULL)
			{
				RFIntHostApp.pF_ErrorPrintf("RFAPP: Fail create event group: %d", inst);
			}
		}

		//	Create a pipe for messages
		//if (NU_Create_Pipe(&pTask->pipe, "PRFAPP", pTask->pipebuf, sizeof(pTask->pipebuf), NU_VARIABLE_SIZE, RFAPP_PIPE_MAX_MSG_LEN, NU_FIFO) != NU_SUCCESS)	{
	   	//		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);    		   
		//}

		//Set options for the message queue
	    sMsgQueueOpt.dwSize = sizeof(sMsgQueueOpt);				//Size of the structure in bytes
	    sMsgQueueOpt.dwFlags = 0;								//Describes the behavior of the message queue
	    sMsgQueueOpt.dwMaxMessages = 0;							//No limit on number of messages to queue
	    sMsgQueueOpt.cbMaxMessage = (RFAPP_PIPE_MAX_MSG_LEN);	//Message size, Maximum number of bytes in each message
	    
		// Write handle to queue.
	    sMsgQueueOpt.bReadAccess = FALSE;
		pTask->hWritePipe = CreateMsgQueue(TEXT("PRFREC"),  &sMsgQueueOpt);
	    
		// Read handle to queue
	    sMsgQueueOpt.bReadAccess = TRUE;
		pTask->hReadPipe = CreateMsgQueue(TEXT("PRFREC"),  &sMsgQueueOpt);
	  
		if (NULL == pTask->hWritePipe || NULL == pTask->hReadPipe)
	    {
			RFIntHostApp.pF_ErrorPrintf("RFAPP: Error create MsgQ!");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
	    } // Create pipe for MlmeNwkInputQueue

		memset((UINT8*) &RFGenReqReply, 0, sizeof(RFGenReqReply));

		// Create a 1-sec generic timer (thread). Thread is in suspended state initially.
		pTask->rappTimerEvent =  CreateEvent(NULL, FALSE, FALSE, NULL);		// to use with timer
		if (!pTask->rappTimerEvent)
		{
			RFIntHostApp.pF_ErrorPrintf("RFAPP: Fail create Timer event!");
		}
		pTask->rappTimerThread = CreateThread(NULL, 0, RFAPP_Timer, pHPRDevice, CREATE_SUSPENDED, NULL);
		if (!pTask->rappTimerThread)
		{
			RFIntHostApp.pF_ErrorPrintf("RFAPP: Fail create Timer Thread!");
        }
		// Set timer thread priority
		CeSetThreadPriority(pTask->rappTimerThread, RF_TIMER_THREAD_PRIORITY);

		//	Create rf application task
		//if(NU_Create_Task(&pTask->task, "TRFAPP", RFAPP_Task, inst, 0, pTask->stack, sizeof(pTask->stack), 
		//	RFAPP_TASK_PRIORITY, 0, NU_PREEMPT, NU_START) != NU_SUCCESS)	{
		//	RFIntHostApp.pF_ErrorHandler("RFAPPTASK: Could not create task!");
		//}

		pTask->task = CreateThread(NULL, 0, RFAPP_Task, pHPRDevice, 0, NULL);
		if (pTask->task == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("RFAPP: Error create RFAPP_Task thread!");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
		}
		// Set thread priority
		CeSetThreadPriority(pTask->task, RFAPP_TASK_PRIORITY); 
#endif
		/* Zero the events before we go to set them. */
//		memset(pTask->hEvents, 0, ARRAY_SIZE(pTask->hEvents));
//		for (inst = 0; inst < RFAPP_EVENT_COUNT; inst++)
//		{
//			//pTask->hEvents[inst] = CreateEvent(NULL, FALSE, FALSE, NULL);
//			OsCreateEvent(&pTask->hEvents[inst]);
//			if (pTask->hEvents[inst] == NULL)
//			{
//				RFIntHostApp.pF_ErrorPrintf("RFAPP: Fail create event group: %d", inst);
//			}
//		}
		//RFLink.state = RFLINKREQ_IDLE_STATE;
		pTask->event_group = OsCreateEventGroup();
		if(!pTask->event_group) {
		    RFIntHostApp.pF_ErrorPrintf("RFAPP: Fail create Timer Thread!");
		}

		//TODO: Verify that queue is created large enough.
	    pTask->hPipe = OsQueueCreate(
	    		RFAPP_TASK_PIPE_BUFFER_SIZE/sizeof(RFAPP_PIPE_MAX_MSG_LEN),
				sizeof(RFAPP_PIPE_MAX_MSG_LEN));

	    if (!pTask->hPipe)
	    {
			RFIntHostApp.pF_ErrorPrintf("RFAPP: Error create MsgQ!");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
	    } // Create pipe for MlmeNwkInputQueue

		memset((UINT8*) &RFGenReqReply, 0, sizeof(RFGenReqReply));

		//pTask->rappTimerThread = OsCreateNamedAdvTaskPri()
		//TODO: Verify this isn't a problem.
		//App timer initialized inside of task now instead. Done to lessen startup races.

		pTask->task = OsCreateNamedAdvTaskPri(RFAPP_Task, 0, 0, RFAPP_TASK_PRIORITY, "ERAppTask");
		//pTask->task = CreateThread(NULL, 0, RFAPP_Task, pHPRDevice, 0, NULL);
		if (!pTask->task)
		{
			RFIntHostApp.pF_ErrorPrintf("RFAPP: Error create RFAPP_Task thread!");
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
		}
	#ifndef	RF_NETWORK_MASTER_DEVICE
		//	Initialize device information strings
		RFAPP_DeviceDataInfoInit();

    //Init. Roaming task
	RFRM_Init();
    #endif
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//	Link request packet post function 
/////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * \author      Hazrat Shah	
 *
 * \date        11/01/2005
 *
 * \return      UINT16
 *
 * \retval      
 *
 * \brief		prepare and post the link request 
 *
 * \param       UINT8 index, UINT8 linkcondition, UINT16 randomno, BOOL broadcast, UINT8 repeats
 *
 * \note
 *
 * \warning
 *
 */
UINT16	RFAPP_PrepareAndSubmitLinkReqPkttoGateway (UINT8 index, UINT8 linkcondition, UINT16 randomno, BOOL broadcast, UINT8 repeats)	{
	UINT16	panid;
	BOOL	valid;
	UINT8	uid[UID_MAX_SIZE];
	INT8	idstr[RF_DEVICE_MAX_CRESNET_ID_STRING_LEN];
	const   TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
	UINT8	len;
	UINT8	devopts;
//	UINT16	powlevel;
//	UINT8	rfid;
//	UINT32	tsid;
	UINT8	flags = (RFAPP_AcquireStatus()?ACQUIRE_ACTIVE:0)|(RFAcquire.acquiresuccessful?ACQUIRE_SUCCESSFUL:0);
//	UINT8	i;
//	UINT16  res;

	(pHF->pF_GetLocalDeviceUID_PANID) (&valid, uid, &panid, (UINT8 *)SLAVE_DEVICE_DEFAULT_UID, SLAVE_DEVICE_DEFAULT_PANID);	
	(pHF->pF_GetIDString)(idstr, &len);
	if (len >= RF_DEVICE_MAX_CRESNET_ID_STRING_LEN)	{	len = RF_DEVICE_MAX_CRESNET_ID_STRING_LEN-1;	}		
	idstr[len] = 0;
	(pHF->pF_GetDeviceOptions)(&devopts);

	return (RFTransmit_BuildLinkRequestPacketAndPostToPipeByIndex (index, flags, linkcondition, devopts, randomno, idstr, broadcast, repeats));	
}

/////////////////////////////////////////////////////////////////////////////////////////////////
//	Acquire functions
/////////////////////////////////////////////////////////////////////////////////////////////////

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Get next channel number	    
 * 
 * \detail 
 *
 * \param		UINT8 channel
 *
 * \return      UINT8 channel
 *
 * \retval      next channel number 	 
 *
 */
UINT8	RFAPP_GetNextChannel (UINT8 channel)	{
	channel++;	
	if ((channel < gLogicalChannel11_c) || (channel > gLogicalChannel26_c))	{
		channel = gLogicalChannel11_c;
	}	
				
	return (channel);
}

#ifndef	RF_NETWORK_MASTER_DEVICE

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Add gateway information that was acquired to the list    
 * 
 * \detail      
 *
 * \return      UINT8 *pname, UINT8 *puid, UINT16 panid, UINT8 channel
 *
 * \retval      None
 *
 */
void	RFAPP_AddToAcquiredList (UINT8 *pname, UINT8 *puid, UINT16 panid, UINT8 channel, UINT8 addrmode)	{	
	UINT32	i;
	UINT8	found;
	
	//	Check to make sure the current entry is not in the list
	for (i=0, found=FALSE; i<RFAcquire.noofGWinAcquire; i++)	{
		if (memcmp((void*)pname, (void*)&RFAcquire.GWAcq[i].name, sizeof(RFAcquire.GWAcq[0].name)) &&
			memcmp((void*)puid, (void*)&RFAcquire.GWAcq[i].uid,   sizeof(RFAcquire.GWAcq[0].uid))  &&
			(RFAcquire.GWAcq[i].panid == panid))	{
			found	= TRUE;	
			break;
		}
	}

	//	Add entry to the list
	if (found == FALSE)	{
		if (RFAcquire.noofGWinAcquire < MAX_GATEWAY_IN_ACQUIRE_REC_SIZE)	{
			memcpy((void*) &RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].name, (void*) pname, sizeof(RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].name));
			memcpy((void*) &RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].uid,  (void*) puid,  sizeof(RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].uid));
			RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].panid   = panid;
			RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].channel = channel;
			RFAcquire.GWAcq[RFAcquire.noofGWinAcquire].addrmode = addrmode;
			RFAcquire.noofGWinAcquire++;
		}	else	{			
			RFIntHostApp.pF_ErrorHandler ("Acquired gateway list is full\r\n");
		}
	}	
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Acquire process handler    
 * 
 * \detail      
 *
 * \return      UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len
 *
 * \retval      None
 *
 */
void RFAPP_AcquireHandler(UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len)	{
	HPRF_UNALIGNED TYPE_RF_LINK_REPLY_PACKET_DATA *preply = (TYPE_RF_LINK_REPLY_PACKET_DATA*) pBuf; 
	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
	BOOL	isTraceEnabled;
	INT8	buf[100];
	BOOL	waitforto;
	
	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	//HPRF_NTOHS(preply->panid);  -- moved below as the function called recursively
	//HPRF_NTOHL(preply->tsid);   -- moved below as the function called recursively

	switch (RFAcquire.state)	{
		case RFACQUIRE_IDLE_STATE:
			//	idle state, acquire process is stopped
			break;

		case RFACQUIRE_INIT_STATE:
			//	Initialize acquire parameters			
			RFAcquire.acquireattempts	= 0;
			RFAcquire.state				= RFACQUIRE_REPEAT_STATE;
			RFAcquire.timer				= 1;
			RFAcquire.noofGWinAcquire	= 0;
			break;

		case RFACQUIRE_REPEAT_STATE:
			RFAcquire.channel = RFReceive_GetLocalDeviceChannel();
			//	continue to the next state
		case RFACQUIRE_SWITCH_CHANNEL_STATE:
			//	Switch channel  
			RFTransmit_BuildChannelSelectionFrameAndPostToPipe (RFAcquire.channel);
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("Acquire switch channel state channel[%d]\r\n", RFAcquire.channel);
			}
			
			//OS_Sleep(4+RFAcquire.acquireattempts);
			/*TODO: Is this supposed to be milliseconds? */
			HwDelayMsec(4+RFAcquire.acquireattempts);

			//	continue to the next state
		case RFACQUIRE_LINKREQ_STATE:
			//	define entry in the pending packet req queue so, the reply gets routed to this function	
			RFAPP_AddEntryofGivenRFPendingPacketReq ((void*)RFAPP_AcquireHandler, RF_PACKET_LINK_REQUEST, 0, RF_PACKET_LINK_REPLY);
			//	set device descriptive entry to temporary access  	
			RFDesc_LinkStateTransition (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_DEVICE_LINK_STATE_EMPTY);		
			RFDesc_LinkStateTransition (RFDesc_GetNetworkServicesDescriptiveTableIndex(), RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);		

			//	Post link request packet			
			RFAcquire.randomno = RFAPP_GetRandomNum();
			RFAPP_PrepareAndSubmitLinkReqPkttoGateway (RFDesc_GetNetworkServicesDescriptiveTableIndex(), RF_PACKET_LINK_ONLY_IF_GATEWAY_IN_ACQUIRE, RFAcquire.randomno, TRUE, 2+RFAcquire.acquireattempts);

			//	Program timer to wait for link reply
			RFAcquire.timer = RFACQUIRE_LINKREQ_REPLY_DELAY+(RFAcquire.acquireattempts/2);
			if (RFAcquire.timer > RFACQUIRE_LINKREQ_REPLY_MAX_DELAY)	{
				RFAcquire.timer	= RFACQUIRE_LINKREQ_REPLY_MAX_DELAY;
			}
			RFAcquire.state = RFACQUIRE_LINKREQ_REPLY_STATE;				
			break;

		case RFACQUIRE_LINKREQ_REPLY_STATE:
			waitforto = FALSE;
			if (trigsrc == RFREQPKT_TRIGSRC_TIMER_EXPIRED)	{
				//	gateway response timed-out  
			}	else if ( ((preply->linkresultcode == RF_PACKET_LINK_SUCCESSFUL_NEW_ENTRY)			||
						   (preply->linkresultcode == RF_PACKET_LINK_SUCCESSFUL_EXISTING_ENTRY))	&&
						   (preply->randomno == RFAcquire.randomno))	{
			    HPRF_NTOHS(preply->panid);
				HPRF_NTOHL(preply->tsid);
				if (isTraceEnabled)	{
					RFIntHostApp.pF_ConsolePrintf ("Acquire link reply received gatewayname[%s] uid[%s] panid[%04x]\r\n", (UINT8*)&preply->gatewayname, ConvertUIDtoString ((UINT8*)&preply->uid, buf, gAddrModeLong_c), preply->panid);
				}
				//	gateway found, store gateway info 
				RFAPP_AddToAcquiredList ((UINT8*)&preply->gatewayname, (UINT8*) &preply->uid, preply->panid, RFAcquire.channel, preply->addrmode);				
				//	submit gateway response to the host
				if (SUCCESS == (*pHF->pF_AcquireUpdate)((TYPE_GATEWAY_IN_ACQUIRE_LIST*)&RFAcquire.GWAcq, RFAcquire.noofGWinAcquire, TRUE))	{
					RFAcquire.acquiresuccessful = TRUE;
				}
				if (!RFAPP_AcquireStatus())	{
					//	acquire was terminated, exit 
					return;					
				}
				
			}	else	{
				//	gateway reply received but, connection was rejected 
				//	(the reponse could be from the device in the previous channel, it was received late)
				if (isTraceEnabled)	{
					RFIntHostApp.pF_ConsolePrintf("Acquire link rejected errorcode[%02x] txrandomno[%04x] rxrandomno[%04x]\r\n", preply->linkresultcode, RFAcquire.randomno, preply->randomno);
				}
				if (RFAcquire.acquireattempts == 0)	{
					RFIntHostApp.pF_ErrorPrintf("RFAcquireError: this gateway rejected the acquire request, rejection reason code[%02x]\r\n", preply->linkresultcode); 	
				}
				waitforto = TRUE;
				if (RFAcquire.timer < 2)	{	RFAcquire.timer++;	}
			}
			
			if (!waitforto)	
			{	
				if( !g_roamingEnabled || RFDesc_RoamingDescriptiveTableIsEmpty() )
				{
					RFAcquire.channel = RFAPP_GetNextChannel (RFAcquire.channel);
				}

				if (RFReceive_GetLocalDeviceChannel() == RFAcquire.channel)	
				{
					//	start next loop start from currently select channel
					RFAcquire.state = RFACQUIRE_REPEAT_STATE;	
					//RFAcquire.timer = 1;
					RFAcquire.acquireattempts++;
				}	
				else	
				{	
					//	try the next channel
					RFAcquire.state = RFACQUIRE_SWITCH_CHANNEL_STATE;	
					//RFAcquire.timer = 1;
				}	
				//	recursive call 
				RFAPP_AcquireHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, (UINT8*) buf, 0);
			}				
			break;
	}
}
#endif

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Acquire process timeout handler    
 * 
 * \detail      
 *
 * \return      None
 *
 * \retval      None
 *
 */
void RFAPP_AcquireHandlerTimer(void)	{	
	UINT8	b=0;
	const	TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
	BOOL	isTraceEnabled;
	
	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	if (RFAcquire.state != RFACQUIRE_IDLE_STATE)	{
		if (RFAcquire.processtimer)	{
			if (--RFAcquire.processtimer == 0)	{
				//	Acquire process timed out, 
				if (isTraceEnabled)	{
					RFIntHostApp.pF_ConsolePrintf("Acquire timed out\r\n");
				}
	#ifdef	RF_NETWORK_MASTER_DEVICE
				//	stop acquire process
				RFAPP_AcquireStop(FALSE);
	#else	
				// inform host about the acquire process result 
				if (SUCCESS == (*pHF->pF_AcquireUpdate)((TYPE_GATEWAY_IN_ACQUIRE_LIST*)&RFAcquire.GWAcq, RFAcquire.noofGWinAcquire, TRUE))	{
					RFAcquire.acquiresuccessful = TRUE;
				}	
	#endif					
			}		
		}
	#ifdef	RF_NETWORK_MASTER_DEVICE
		//	net master
	#else
		if (RFAcquire.timer)	{	
			if (--RFAcquire.timer == 0)	{
				RFAPP_AcquireHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);			
			}	
		}	
	#endif
	}
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Start the Acquire process    
 * 
 * \detail      
 *
 * \param		UINT32 timeout
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 *
 */
UINT16 RFAPP_AcquireStart (UINT32 timeout)	{
	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
	
	HPRF_DEBUG(pr_info("===== %s ===== RFAcquire.state = %u\n", __FUNCTION__, RFAcquire.state);)
	if (RFAcquire.state == RFACQUIRE_IDLE_STATE)	{
		//	clear acquire parameters 
		memset ((INT8*)&RFAcquire, 0, sizeof(TYPE_RF_ACQUIRE_REC));
		//	setup acquire process timeout
		RFAcquire.processtimer = (timeout<RFACQUIRE_PROCESS_MAX_TIMEOUT)?(timeout*60):(RFACQUIRE_PROCESS_MAX_TIMEOUT*60);
	#ifdef	RF_NETWORK_MASTER_DEVICE
		pHF->pF_AcquireStartIndication();
		RFAcquire.state = RFACQUIRE_INIT_STATE;
		RFAcquire.timer = 0;
	#else		
		//	Clear control system/gateway connect status  
		RFReceive_SetGatewayConnectStatus (FALSE);
		//	Stop link request handler		
		RFAPP_LinkRequestHandlerStop();
		//	Start acquire handler		
		RFAcquire.state = RFACQUIRE_INIT_STATE;
		RFAcquire.timer = 1;
		//	delete entry in the pending packet req queue 	
		RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*)RFAPP_AcquireHandler, RF_PACKET_ACTIVE_SCAN_REQUEST, 0);
		RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*)RFAPP_AcquireHandler, RF_PACKET_LINK_REQUEST,		0);
		if(g_roamingEnabled)
        {
			RFDesc_ReleaseEntryInRoamingDescriptiveTable(g_acquireSlot-1);
//			RFIntHostApp.pF_SetRoamingGatewayUID_PANID(0,0);//this will update param.rfparam...
		}
	#endif
	}	else	{
		//	retrigger the acquire timeout
		RFAcquire.processtimer = (timeout<RFACQUIRE_PROCESS_MAX_TIMEOUT)?(timeout*60):(RFACQUIRE_PROCESS_MAX_TIMEOUT*60);
	}

	return (SUCCESS);
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Stop the Acquire process, restart network if requested   
 * 
 * \detail      
 *
 * \param		BOOL ReStartNetwork	
 *
 * \return      UINT16
 *
 * \retval      SUCCESS/FAILURE
 *
 */
UINT16 RFAPP_AcquireStop (BOOL ReStartNetwork)	{
	UINT16 res = FAILURE;
	const TYPE_INTERFACE_TO_RFAPP_APP *pRFA = &RFIntRFApp;
	const TYPE_INTERFACE_TO_HOST_APP  *pHF = &RFIntHostApp;

	HPRF_DEBUG(DmConsolePrintf("%s - %p %p\n", __FUNCTION__, pRFA, pHF);)
	if (RFAcquire.state != RFACQUIRE_IDLE_STATE)	{
		RFAcquire.state = RFACQUIRE_IDLE_STATE;
		RFAcquire.timer = 0;	
		RFAcquire.processtimer = 0;

	#ifdef	RF_NETWORK_MASTER_DEVICE	
		pHF->pF_AcquireStopIndication();		
	#else	
		if (ReStartNetwork)	{
		    DmConsolePrintf("%s - RestartNetwork\n", __FUNCTION__);
			//	Restart network, start link request process 
			pRFA->pF_LookupSettingsAndConnectDeviceToNetwork();	
		}
        DmConsolePrintf("%s - DeleteEntryofGivenRFPendingPacketReq\n", __FUNCTION__);

		//	delete entry in the pending packet req queue 	
		RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*)RFAPP_AcquireHandler, RF_PACKET_LINK_REQUEST,		0);
	
	#endif	
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Get Acquire process running status    
 * 
 * \detail      
 *
 * \param		None	
 *
 * \return		BOOL
 *
 * \retval      TRUE/FALSE
 *
 */
BOOL	RFAPP_AcquireStatus (void)	{
	return ((RFAcquire.state!=RFACQUIRE_IDLE_STATE)?TRUE:FALSE);
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Get acquire remaining time      
 * 
 * \detail      
 *
 * \param		None
 *
 * \return      UINT32
 *
 * \retval      
 *
 */
UINT32	RFAPP_AcquireRemainingTime (void)	{
	UINT32 seconds = 0;		

	if (RFAcquire.state)	{		
		seconds = RFAcquire.processtimer;	
	}
	return (seconds);
}


////////////////////////////////////////////////////////////////////////////////////////////////
//	Link request handler functions
////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef	RF_NETWORK_MASTER_DEVICE

void RFAPP_LinkRequestHandlerTimer(void)	{	
}

#else

void RFAPP_ProcessLinkReply (UINT8 *pBuf, UINT16 len)	{
	TYPE_RF_LINK_REPLY_PACKET_DATA *preply = (TYPE_RF_LINK_REPLY_PACKET_DATA*) pBuf; 
	BOOL isTraceEnabled;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	//	A reply was received, process it
	if ((preply->linkresultcode == RF_PACKET_LINK_REJECTED_DEVICE_NOT_LINKED) && RFReceive_IsGatewayConnected() && (!RFAPP_AcquireStatus()) ) {
		RFDesc_LinkStateTransition (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_DEVICE_LINK_STATE_FAIL);
		if (!RFAPP_AcquireStatus())	{
			RFDesc_LinkStateTransition (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_DEVICE_LINK_STATE_CONNECTING);
		} 
		RFAPP_ConnectionFailedwithGateway();
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf("LinkReply Error: Devices needs to be linked\r\n");
		}
	}
}

/**
 *
 * \date        09/20/2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Get link handler delay     
 * 
 * \detail      
 *
 * \param		UINT8 attempt
 *
 * \return      UINT8 delay time	
 *
 * \retval      
 *
 */
UINT8	RFAPP_GetLinkHandlerDelay (UINT8 attempts)	{
	UINT8	dly;
	if (attempts >= RFLINKREQ_MAX_DELAY_CNT)	{	attempts = RFLINKREQ_MAX_DELAY_CNT-1;	}	
	dly = RFLinkReqDelays[attempts] + RFReceiver_GetTotalRetryDlyTimeInSeconds();
	return (dly?dly:1);	
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Process Link rely     
 * 
 * \detail      
 *
 * \param		UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len
 *
 * \return      void	
 *
 * \retval      
 *
 */
void RFAPP_LinkRequestHandler(UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len)	{
	HPRF_UNALIGNED TYPE_RF_LINK_REPLY_PACKET_DATA *preply = (TYPE_RF_LINK_REPLY_PACKET_DATA*) pBuf; 
	//UINT32	i;
	BOOL	isTraceEnabled;
	UINT8	b /*, lq, lql, lqr */;
	UINT8 indexInRoamingTable;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	//HPRF_NTOHS(preply->panid); -- moved below as we have recursive calls.
	//HPRF_NTOHL(preply->panid); -- moved below as we have recursive calls.
    
	switch (RFLink.state)	{
		case RFLINKREQ_IDLE_STATE:
			//	idle state, link req handler is stopped
			break;

		case RFLINKREQ_REQREPLY_INIT_STATE:
			//	Prepare and issue the link request packet
			//RETAILMSG(1, (L"REQREPLY_INIT_STATE, time %d", GetTickCount()));
			RFLink.attempts = 0;
			RFLink.prequeryattempts = 0;

		case RFLINKREQ_REQUEST_CHKLQI_STATE:
			if (RFLink.prequeryattempts >= RFReceive_GetLocalDevicePreLinkQueryPacketCnt())	{
				//RETAILMSG(1, (L"REQUEST_CHKLQI_STATE, -if-,time %d", GetTickCount()));
				RFLink.timer    = 1;		
				RFLink.state    = RFLINKREQ_REQUEST_STATE;
				RFAPP_LinkRequestHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);
			}	else	{
				//	define entry in the pending packet req queue so, the reply gets routed to this function	
				RFAPP_AddEntryofGivenRFPendingPacketReq ((void*)RFAPP_LinkRequestHandler, RF_PACKET_LINK_REQUEST, 0, RF_PACKET_LINK_REPLY);
				//	post link query packet
				RFLink.randomno = RFAPP_GetRandomNum();
				//RETAILMSG(1, (L"REQUEST_CHKLQI_STATE, -else-, time %d, random no: %x", GetTickCount(), RFLink.randomno));
				RFLink.retryhighlevel = g_pRFDeviceDescriptiveTable[DEVICE_DESCRIPTIVE_TABLE_INDEX].COMLog.txpacketretrycount;
				RFLink.retrylowlevel  = Rfdiags_getLowLevelRetryPacketCnt();
				RFAPP_PrepareAndSubmitLinkReqPkttoGateway (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_PACKET_LINK_QUERY, RFLink.randomno, FALSE, 0);			
				//	program wait time
				RFLink.timer    = RFAPP_GetLinkHandlerDelay(RFLink.attempts);	
				//	Set up link reply handler
				RFLink.state    = RFLINKREQ_REQUEST_LQIREPLYWAIT_STATE;
			}	
			break;

		// HS - 11/02/07 (added back in) does not make use of LQI level or retry/CCA errors
		//		used to make sure a packet can be exchanged with GW prior to issuing the link request
		//		reducing the probabilty of premature reconnect in the fringe region
		case RFLINKREQ_REQUEST_LQIREPLYWAIT_STATE:
			//	delete entry from the pending packet req queue
			RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*)RFAPP_LinkRequestHandler, RF_PACKET_LINK_REQUEST, 0);
			if (trigsrc == RFREQPKT_TRIGSRC_TIMER_EXPIRED)	{
				//	Reply timed out
				RFLink.attempts++;											
				RFLink.timer = 1;
				RFLink.state = RFLINKREQ_REQUEST_CHKLQI_STATE;
				//RETAILMSG(1, (L"REQUEST_LQIREPLYWAIT_STATE timeout, time %d, attempts %d", GetTickCount(), RFLink.attempts));
				if (isTraceEnabled)	{
					//	Link reply wait timeout 
					RFIntHostApp.pF_ConsolePrintf("RFLinkQueryError: query reply wait timeout occured\r\n");
				}
				//	HS - 6/22/07 
				//	Reset the RFPHY chip: 
				//	The RF PHY chip enters a mode where it sends invalid packet data whenever
				//	a packet is posted to it for transmission. To get the chip out of this mode
				//	it needs to be reset.
				RFTransmit_BuildResetMACMessageAndPostToPipe(FALSE);
				RFLink.prequeryattempts = 0;
			}	else	{
				//	A reply was received, process it
				//RETAILMSG(1, (L"REQUEST_LQIREPLYWAIT_STATE reply rx, time %d, result code %d", GetTickCount(), preply->linkresultcode));
				if (preply->linkresultcode == RF_PACKET_LINK_QUERY_REPLY)	{			
					//	check the LQI value, if higher than threshold procceed with connection attempts
					//	enhancement: detection and recover based on dynamic thr 
					//RFDesc_GetLinkQualityOfLocalDevice  (DEVICE_DESCRIPTIVE_TABLE_INDEX, &lql);	
					//RFDesc_GetLinkQualityOfRemoteDevice (DEVICE_DESCRIPTIVE_TABLE_INDEX, &lqr);						
					//lq = (lql > lqr)?lql:lqr;
					//if (lq < RFReceive_GetLocalDeviceRFLinkLQIThr())	{						
						//if ((g_pRFDeviceDescriptiveTable[DEVICE_DESCRIPTIVE_TABLE_INDEX].COMLog.txpacketretrycount - RFLink.retryhighlevel)  == 0)	{
						RFLink.prequeryattempts++;
						RFLink.state = RFLINKREQ_REQUEST_CHKLQI_STATE;
						RFLink.timer = 0;
						RFLink.attempts = 0;
						//	link device to gw
						RFAPP_LinkRequestHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);
					//}	else	{
					//	RFLink.state = RFLINKREQ_REQUEST_CHKLQI_STATE;
					//	RFLink.timer = 1;	
					//}
				}	else	{
					if (isTraceEnabled)	{
						//	Link request was rejected
						RFIntHostApp.pF_ConsolePrintf("RFLinkQueryError: Gateway rejected the rf link query, errorcode = %02x \r\n", preply->linkresultcode);
						RFIntHostApp.pF_ConsolePrintf("             Received random no %04x, expected random no %04x\r\n", preply->randomno, RFLink.randomno);
					}
					//	Try again in x-seconds	
					RFLink.timer = RFAPP_GetLinkHandlerDelay(RFLink.attempts);
					RFLink.attempts++;
					RFLink.state = RFLINKREQ_REQUEST_CHKLQI_STATE;
				}			
			}
			break;

		case RFLINKREQ_REQUEST_STATE:
			//	define entry in the pending packet req queue so, the reply gets routed to this function	
			RFAPP_AddEntryofGivenRFPendingPacketReq ((void*)RFAPP_LinkRequestHandler, RF_PACKET_LINK_REQUEST, 0, RF_PACKET_LINK_REPLY);
			//	post link request packet
			RFLink.randomno = RFAPP_GetRandomNum();
			RFAPP_PrepareAndSubmitLinkReqPkttoGateway (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_PACKET_LINK_UNCONDITIONAL, RFLink.randomno, FALSE, 0);			
			RFLink.timer    = RFAPP_GetLinkHandlerDelay(RFLink.attempts);	
			//RETAILMSG(1, (L"RREQUEST_STATE, time %d, random no %d, timer: %d", GetTickCount(), RFLink.randomno, RFLink.timer));
			//	Set up link reply handler
			RFLink.state    = RFLINKREQ_REPLYWAIT_STATE;
			break;

		case RFLINKREQ_REPLYWAIT_STATE:	
			//	delete entry from the pending packet req queue
			RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*)RFAPP_LinkRequestHandler, RF_PACKET_LINK_REQUEST, 0);
			if (trigsrc == RFREQPKT_TRIGSRC_TIMER_EXPIRED)	{
				//	Reply timed out
				RFLink.attempts++;											
				RFLink.timer = 1;
				//RETAILMSG(1, (L"RFLINKREQ_REPLYWAIT_STATE: Timeout. Time %d, attempts %d", GetTickCount(), RFLink.attempts));
				RFLink.state = RFLINKREQ_REQUEST_STATE;
				if (isTraceEnabled)	{
					//	Link reply wait timeout 
					RFIntHostApp.pF_ConsolePrintf("RFLinkError: Link reply wait timeout occured\r\n");
				}
				//	HS - 6/22/07 
				//	Reset the RFPHY chip: 
				//	The RF PHY chip enters a mode where it sends invalid packet data whenever
				//	a packet is posted to it for transmission. To get the chip out of this mode
				//	it needs to be reset.
				RFTransmit_BuildResetMACMessageAndPostToPipe(FALSE);	
			}	else	{
				//RETAILMSG(1, (L"RFLINKREQ_REPLYWAIT_STATE: Reply Rx. Time %d, result code: %d, Random No: %d, Rxed Random No: %d", GetTickCount(), preply->linkresultcode, RFLink.randomno, preply->randomno));
				//	A reply was received, process it
				if ( ((preply->linkresultcode == RF_PACKET_LINK_SUCCESSFUL_NEW_ENTRY) ||
					  (preply->linkresultcode == RF_PACKET_LINK_SUCCESSFUL_EXISTING_ENTRY)) &&
					  (preply->randomno == RFLink.randomno))	{			
						
					//	(05/25/07 - HS) Set link state to active in the device decriptive entry (connection succefully established)
					RFDesc_LinkStateTransition (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_DEVICE_LINK_STATE_ACTIVE);
					HPRF_NTOHS(preply->panid);
					HPRF_NTOHL(preply->tsid);
					//	Save device id string
					RFDesc_UpdateDeviceIDInfo(DEVICE_DESCRIPTIVE_TABLE_INDEX, preply->power, preply->tsid, preply->cnetid, 
																			  preply->panid, 0, preply->gatewayname, 
																			  strlen((INT8*)preply->gatewayname));	
					
					//	Link request was successful, set gateway status to connected
					RFReceive_SetGatewayConnectStatus(TRUE);
					//	Update control system connect status
					RFReceive_SetControlSystemConnectedStatus ((RFRecMsg_GetOptionFlags()&RF_HEADER_EXPANSION_CS_CONNECT_STATE)?TRUE:FALSE, TRUE);
					RFReceive_SetControlSystemConnectedNetTypeCresnet ((RFRecMsg_GetOptionFlags()&RF_HEADER_EXPANSION_CRESNET_NET)?TRUE:FALSE);
					RFReceive_SetControlSystemConnectedNetTypeEthernet((RFRecMsg_GetOptionFlags()&RF_HEADER_EXPANSION_ETHERNET_NET)?TRUE:FALSE);
					//	Clear device statistic counters, a new connection was established 
					//	RFDesc_SetSlaveDeviceLinkEstablishedParameters(DEVICE_DESCRIPTIVE_TABLE_INDEX);
					//	Clear timer, set handler state to idle
					RFLink.state = RFLINKREQ_IDLE_STATE;
					RFLink.timer = 0;
					//	Clear the device was just acquired flag, after the link reply is received	
					//	This flag is set right after the device is acquired successfully.  
					RFAcquire.acquiresuccessful = FALSE;
					//RETAILMSG(1, (L"LINK SUCCESS %d!!", GetTickCount()));
					if(g_roamingEnabled)
					{
						if(RFDesc_GetIndexToRoamingDescriptiveTableByUID(g_pRFDeviceDescriptiveTable[0].IDInfo.uid,&indexInRoamingTable))
							RFRecMsg_FillLQ_Buffer(indexInRoamingTable,g_pRFDeviceDescriptiveTable[0].LQ.linkqualityAtLocalDevice);
					}
				}	else	{
					if (isTraceEnabled)	{
						//	Link request was rejected
						RFIntHostApp.pF_ConsolePrintf("RFLinkError: Gateway rejected the rf link request, errorcode = %02x \r\n", preply->linkresultcode);
						RFIntHostApp.pF_ConsolePrintf("             Received random no %04x, expected random no %04x\r\n", preply->randomno, RFLink.randomno);
					}
					//RETAILMSG(1, (L"TRY AGAIN in x seconds: time %d", GetTickCount()));
					//	Try again in x-seconds	
					RFLink.timer = RFAPP_GetLinkHandlerDelay(RFLink.attempts);	
					//	Increase link request reject counter
					RFLink.attempts++;
					RFLink.state = RFLINKREQ_REQUEST_STATE;
					if (RFLink.attempts == 1)	{
						if (isTraceEnabled)	{
							RFIntHostApp.pF_ErrorPrintf("RFLinkError: Gateway rejected the rf link request, rejection reason code[%02x]\r\n", preply->linkresultcode); 	
						}	
					}
				}
			}			
			break;				
	}
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       RF Link Process timeout      
 * 
 * \detail      
 *
 * \param		void	
 *
 * \return      void	
 *
 * \retval      
 *
 */
void RFAPP_LinkRequestHandlerTimer(void)	{	
	UINT8	b = 0;

	if (RFLink.state != RFLINKREQ_IDLE_STATE)	{
		if (RFLink.timer)	{	
			if (--RFLink.timer == 0)	{
				RFAPP_LinkRequestHandler (RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);			
			}	
		}	
	}
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Start Link process     
 * 
 * \detail      
 *
 * \param		UINT8 delay	
 *
 * \return      void	
 *
 * \retval      
 *
 */
void RFAPP_LinkRequestHandlerStart(UINT8 delay)	{	
	if ((!RFAPP_AcquireStatus()) && (RFLink.state == RFLINKREQ_IDLE_STATE))	{
		//	Clear control system and gateway connections
		RFReceive_SetGatewayConnectStatus (FALSE);			
		RFLink.attempts = 0;
		RFLink.state = RFLINKREQ_REQREPLY_INIT_STATE;	
		RFLink.timer = (delay?delay:1);		//	dly value of 0 is not allowed
	}
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       This function is called when connection with the gateway fails     
 * 
 * \detail      
 *
 * \param		void
 *
 * \return      void	
 *
 * \retval      
 *
 */
void	RFAPP_ConnectionFailedwithGateway (void)	{
	//	Ignore communication failures while acquire is running, the acquire process deals with the communications errors 	
	RFAPP_LinkRequestHandlerStart(RFAPP_GetRandomNum()&0x3);		
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       This function is called to establish connection with the gateway      
 * 
 * \detail      
 *
 * \param		UINT8 *pgatewayuid, UINT16 gatewaypanid, UINT8 channel
 *
 * \return      UINT16	
 *
 * \retval      SUCCESS/FAILURE
 *
 */
UINT16	RFAPP_EstablishConnectionwithGateway (UINT8 *pgatewayuid, UINT16 gatewaypanid, UINT8 channel, UINT8 gatewayaddrmode)	{
	//	Important note:
	//	Prior to making this call the device UID, PANID and default channel were set to 
	//	initialize the stack. The stack only accepts UID during initialization
	//	This call sets the gateway uid/panid parameters

	RFReceive_SetLocalDeviceChannel (channel);

	//	set device descriptive entry to attempting to establish connection state  
	RFDesc_LinkStateTransition (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_DEVICE_LINK_STATE_FAIL);
	RFDesc_SetUID	(DEVICE_DESCRIPTIVE_TABLE_INDEX, pgatewayuid);
	RFDesc_SetPanID (DEVICE_DESCRIPTIVE_TABLE_INDEX, gatewaypanid);
	RFDesc_SetAddrMode (DEVICE_DESCRIPTIVE_TABLE_INDEX, gatewayaddrmode);
	//	set device descriptive entry to attempting to establish connection state  
	RFDesc_LinkStateTransition (DEVICE_DESCRIPTIVE_TABLE_INDEX, RF_DEVICE_LINK_STATE_CONNECTING);
	//	Establish gateway link ASAP
	RFAPP_LinkRequestHandlerStart(1);	

	return (SUCCESS);
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Stop link request process       
 * 
 * \detail      
 *
 * \param		void	
 *
 * \return      void	
 *
 * \retval      
 *
 */
void	RFAPP_LinkRequestHandlerStop (void)	{	
	RFLink.state = RFLINKREQ_IDLE_STATE;
	RFLink.timer = 0;	

	//	delete entry from the pending packet req queue
	RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*)RFAPP_LinkRequestHandler, RF_PACKET_LINK_REQUEST, 0);
}
#endif

////////////////////////////////////////////////////////////////////////////////////////////////
//	RF Echo test request/reply functions  
////////////////////////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function processes request packet replies  	     	 	   	     			           
 *
 * \detail		
 *
 *						
 * \date        04/03/2007
 *					
 * \param	    UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len
 *
 * \return		void  					
 *
 * \retval				
 */
void RFAPP_EchoRequestHandler(UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len)	{
	TYPE_RF_ECHO_REPLY_PACKET_DATA *preply = (TYPE_RF_ECHO_REPLY_PACKET_DATA*) pBuf; 
	UINT8 	index[MAX_DESCRIPTIVE_TABLE_ENTRIES+2];
	UINT16 	indexcnt;	
//	UINT16  i, j;
//	UINT8	buf[300];
	BOOL	isTraceEnabled;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	indexcnt = RFDesc_GetActiveRFDeviceTableIndexByRFID (RFEcho.EchoReq.rfid, index);	

	if (trigsrc == RFREQPKT_TRIGSRC_TIMER_EXPIRED)	{
		//	Echo timeout
		if (isTraceEnabled)	{
			RFIntHostApp.pF_ConsolePrintf (" Echo reply timeout \r\n");
		}
		RFEcho.EchoReply.noresponses++;
	}	else	{
		if (0 != memcmp (RFEcho.EchoReq.buf, preply->data, RFEcho.EchoReq.buflen))	{	RFEcho.EchoReply.invalidresponses++;	}
		if (indexcnt)	{
			//	update link quality average
			RFDesc_GetLinkQualityOfLocalDevice (index[0], &RFEcho.EchoReply.localrxlqi);
			//RFDesc_GetLinkQualityOfRemoteDevice(index[0], &RFEcho.EchoReply.remoterxlqi);
			RFEcho.EchoReply.remoterxlqi		   = preply->linkquality;					//	remote device LQI
			RFEcho.EchoReply.avgrxlqi			  += RFEcho.EchoReply.localrxlqi;
			RFEcho.EchoReply.avgremotedevicerxlqi += preply->linkquality;
		}
	}

	if (RFEcho.EchoReq.pcbf)	{	
		if (indexcnt)	{
			pdst = g_pRFDeviceDescriptiveTable+index[0];	
			if (RFEcho.EchoReq.repeats)	{	
				RFEcho.EchoReply.done = FALSE;	
			}	else	{	
				RFEcho.EchoReply.done = TRUE;	
				RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*) RFAPP_EchoRequestHandler, RF_PACKET_ECHO_REQUEST, 0);
				RFEcho.timer = 0;			
				if (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses)	{	
					RFEcho.EchoReply.avgrxlqi /= (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses);	
					RFEcho.EchoReply.avgremotedevicerxlqi /= (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses);
				}				
			}
			RFEcho.EchoReply.tick = OS_RetrieveClock()-RFEcho.starttick;	
			RFEcho.EchoReply.txpacketcount  = pdst->COMLog.txpacketcount-RFEcho.starttxpktcnt;
			RFEcho.EchoReply.txpacketpayloadbytecount = pdst->COMLog.txpacketpayloadbytecount-RFEcho.starttxdatabytecnt;
			RFEcho.EchoReply.rxpacketcount = pdst->COMLog.rxpacketcount-RFEcho.startrxpktcnt;
			RFEcho.EchoReply.rxpacketpayloadbytecount = pdst->COMLog.rxpacketpayloadbytecount-RFEcho.startrxdatabytecnt;					

			if (trigsrc == RFREQPKT_TRIGSRC_TIMER_EXPIRED)	{
#ifdef		FREESCALE_802_15_4_STACK
				RFEcho.EchoReply.localrxlqi  = 0;
				RFEcho.EchoReply.remoterxlqi = 0;
#else
				RFEcho.EchoReply.localrxlqi  = 255;
				RFEcho.EchoReply.remoterxlqi = 255;
#endif
			}	
			(*RFEcho.EchoReq.pcbf) (&RFEcho.EchoReply);
		}	else	{	
			RFEcho.EchoReply.done = TRUE;
#ifdef		FREESCALE_802_15_4_STACK
				RFEcho.EchoReply.localrxlqi  = 0;
				RFEcho.EchoReply.remoterxlqi = 0;
				if (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses)	{	
					RFEcho.EchoReply.avgrxlqi /= (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses);	
					RFEcho.EchoReply.avgremotedevicerxlqi /= (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses);
				}	else	{
					RFEcho.EchoReply.avgrxlqi			  = 0;
					RFEcho.EchoReply.avgremotedevicerxlqi = 0;
				}
#else
				RFEcho.EchoReply.localrxlqi  = 255;
				RFEcho.EchoReply.remoterxlqi = 255;				
				if (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses)	{	
					RFEcho.EchoReply.avgrxlqi /= (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses);	
					RFEcho.EchoReply.avgremotedevicerxlqi /= (RFEcho.EchoReply.repeatno-RFEcho.EchoReply.noresponses);
				}	else	{
					RFEcho.EchoReply.avgrxlqi			  = 255;
					RFEcho.EchoReply.avgremotedevicerxlqi = 255;
				}
#endif
			(*RFEcho.EchoReq.pcbf) (&RFEcho.EchoReply);
			RFEcho.EchoReq.repeats = 0;			
			RFEcho.EchoReply.tick = 0;		
			RFEcho.EchoReply.txpacketcount = 0;
			RFEcho.EchoReply.txpacketpayloadbytecount = 0;
			RFEcho.EchoReply.rxpacketcount = 0;
			RFEcho.EchoReply.rxpacketpayloadbytecount = 0;
			RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*) RFAPP_EchoRequestHandler, RF_PACKET_ECHO_REQUEST, 0);
			RFEcho.timer = 0;	
		}
	}
	
	if (RFEcho.EchoReq.repeats)	{
		RFEcho.timer  = RFAPP_ECHO_REPLY_PACKET_TIMEOUT;
		RFTransmit_BuildEchoRequestPacketAndPostToPipeByRFID (RFEcho.EchoReq.rfid, (UINT8*)&RFEcho.EchoReq.buf, RFEcho.EchoReq.buflen);
		RFEcho.EchoReply.repeatno++;
		if (RFEcho.EchoReq.repeats)	{	RFEcho.EchoReq.repeats--;	}	
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function times out echo packet reply   	     	 	   	     			           
 *
 * \detail		
 *
 *						
 * \date        04/03/2007
 *					
 * \param	    void	
 *
 * \return		void  					
 *
 * \retval				
 */
void RFAPP_EchoRequestHandlerTimer (void)	{	
	if (RFEcho.timer)	{
		if (--RFEcho.timer == 0)	{
			RFAPP_EchoRequestHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, NULL, 0);		
		}			
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function starts/stops the echo test     	     	 	   	     			           
 *
 * \detail		
 *
 *						
 * \date        04/03/2007
 *					
 * \param	    TYPE_ECHO_REQUEST_PACKET *pEchoReq
 *
 * \return		INT32	  					
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFAPP_PostEchoRequest (TYPE_RF_ECHO_REQUEST_PACKET *pEchoReq)	{
	UINT8 	indexbuf[MAX_DESCRIPTIVE_TABLE_ENTRIES+2];
	UINT16 	indexcnt = RFDesc_GetActiveRFDeviceTableIndexByRFID (pEchoReq->rfid, indexbuf);	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;
	INT32	res = SUCCESS;

	if (pEchoReq->start)	{
		if ((SUCCESS == RFAPP_AddEntryofGivenRFPendingPacketReq ((void*)RFAPP_EchoRequestHandler, RF_PACKET_ECHO_REQUEST, 0, RF_PACKET_ECHO_REPLY)) && indexcnt)	{
			pdst = g_pRFDeviceDescriptiveTable+indexbuf[0];
			//	copy the echo requester info
			memcpy ((UINT8*)&RFEcho.EchoReq, (UINT8*)pEchoReq, sizeof(RFEcho.EchoReq));
			//	clear the echo reply parameters
			memset ((UINT8*)&RFEcho.EchoReply, 0, sizeof(RFEcho.EchoReply));
			//	check buffer length
			if (RFEcho.EchoReq.buflen>RF_ECHO_PACKET_DATA_LEN-1)	{	RFEcho.EchoReq.buflen = RF_ECHO_PACKET_DATA_LEN-1;	}		
			//	set echo pkt reply timeout
			RFEcho.timer   = RFAPP_ECHO_REPLY_PACKET_TIMEOUT;
			RFEcho.starttick			= OS_RetrieveClock();
			RFEcho.starttxpktcnt		= pdst->COMLog.txpacketcount;
			RFEcho.starttxdatabytecnt	= pdst->COMLog.txpacketpayloadbytecount;
			RFEcho.startrxpktcnt		= pdst->COMLog.rxpacketcount;
			RFEcho.startrxdatabytecnt	= pdst->COMLog.rxpacketpayloadbytecount;
			RFEcho.startrxdatabytecnt   = pdst->COMLog.rxpacketpayloadbytecount;
			if (RFEcho.EchoReq.repeats)	{	RFEcho.EchoReq.repeats--;	}	
			RFEcho.EchoReply.repeatno++;
			//	Post echo packet <add rfid support eg. broadcast ...)
			RFTransmit_BuildEchoRequestPacketAndPostToPipeByRFID (RFEcho.EchoReq.rfid, (UINT8*)&RFEcho.EchoReq.buf, RFEcho.EchoReq.buflen);	
		}	else	{
			res = FAILURE;
		}
	}	else	{
		//	stop echo test
		RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*) RFAPP_EchoRequestHandler, RF_PACKET_ECHO_REQUEST, 0);
	}
	
	return (res);
}

////////////////////////////////////////////////////////////////////////////////////////////////
//	RF ActiveScan/Energy Scan request/reply functions	  
////////////////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////////////////
//	Post packets to rf application 
////////////////////////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function posts message to rf application task pipe	     	 	   	     			           
 *
 * \detail		
 *
 *						
 * \date        04/03/2007
 *					
 * \param	    UINT8 *pbuf, UINT16 buflen
 *
 * \return		UINT16  					
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16	RFAPP_PostToPipe (UINT8 *pbuf, UINT16 buflen)	{
	UINT16 res = SUCCESS;
	TYPE_RFAPP_TASK* pTask = &RFAppTask;	
#if defined (OS_NUCLEUS)
	if (NU_SUCCESS == NU_Send_To_Pipe(&pTask->pipe, (INT8 *)pbuf, buflen, NU_NO_SUSPEND))	{
		//	raise message event
		NU_Set_Events(&pTask->events, RFAPP_RFPACKET_IN_PIPE_EVENT,  NU_OR);
	}	else	{
		//	error, message could not be posted to the PIPE? 		 				
		RFIntHostApp.pF_ErrorPrintf("PRFAPP: pipe full, reset tx pipe");    	
		//	reset pipe
    	NU_Reset_Pipe(&pTask->pipe);	
		res = FAILURE;		
	}
#elif defined (OS_WINCE)
	if (TRUE == WriteMsgQueue(pTask->hWritePipe, (void*)pbuf, buflen, OS_NO_SUSPEND, 0))
	{
		SetEvent(pTask->hEvents[RFAPP_RFPACKET_IN_PIPE_EVENT]);
	}
	else
	{
		//	error, message could not be posted to the PIPE? 		 				
		RFIntHostApp.pF_ErrorPrintf("PRFAPP: pipe write error: %d", GetLastError()); 
		// reset pipe is not present in wince, and there is no limit on number of messages
		// to post to queue in wince
    	res = FAILURE;		
	}
#endif
	HPRF_QUEUE_DEBUG(DmConsolePrintf("--- %s:%i ---> queueing %lu bytes to hPipe\n", \
            __FUNCTION__, __LINE__, buflen);)
	INT32 bytesWritten = OsQueueBytes(pTask->hPipe, pbuf, buflen);
	if(bytesWritten > 0) {
	    OsSetEvent(pTask->event_group, RFAPP_RFPACKET_IN_PIPE_EVENT);
	}
	else {
	    HPRF_DEBUG(dump_stack();)
	    RFIntHostApp.pF_ErrorPrintf("PRFAPP: pipe write error: buflen=%hu bytesWritten=%i\n", buflen, bytesWritten);
	}
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called when the HPRFGW gets disconnected from Ethernet      	 	   	     			           
 *
 * \detail			 
 *				
 *
 *						
 * \date        04/09/2007
 *					
 * \param	    None
 *
 * \return		None	  					
 *
 * \retval		None
 */
void RFAPP_EthernetCIPDisConnected (void)	{
#ifdef	RF_NETWORK_MASTER_DEVICE
	RFIntHostApp.pF_ConsolePrintf  ("CIP was dis-connected \r\n");
	RFAPP_ControlSystemStatusHandler (0, RFAPP_CIP_OFFLINE);		
	//	Broadcast latest control system connection state event to all slave devices
	//RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST);
#endif
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called when the HPRFGW gets connected to Ethernet      	 	   	     			           
 *
 * \detail			 
 *				
 *
 *						
 * \date        04/09/2007
 *					
 * \param	    None
 *
 * \return		None	  					
 *
 * \retval		None
 */
void RFAPP_EthernetCIPConnected (void)	{
#ifdef	RF_NETWORK_MASTER_DEVICE
	RFIntHostApp.pF_ConsolePrintf ("CIP was connected \r\n");
	RFAPP_ControlSystemStatusHandler (0, RFAPP_CIP_ONLINE);	
	//	Broadcast control system connect event to all slave devices
	//RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST);
#endif
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called when the Cresnet buffer overflows      	 	   	     			           
 *
 * \detail			 
 *				
 *
 *						
 * \date        04/09/2007
 *					
 * \param	    None
 *
 * \return		None	  					
 *
 * \retval		None
 */
void RFAPP_CresnetBufferOverflow (void)	{
#ifdef	RF_NETWORK_MASTER_DEVICE
	//	Post Cresnet transmit buffer overflow event
	TYPE_RFAPP_TASK* pTask = &RFAppTask;
#if defined (OS_NUCLEUS)
	NU_Set_Events(&pTask->events, RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT,  NU_OR);	
#elif defined (OS_WINCE)
	SetEvent(pTask->hEvents[RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT]);
#endif
	OsSetEvent(RFAppTask.event_group, RFAPP_CRESNET_TXPIPE_OVERFLOW_EVENT);
#endif
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief		device data info string initialization, the data is copied to the image in stack
 *				the string crc is also, calculated and stored
 *	
 * \param       void
 *
 * \note
 *
 * \warning
 *
 */
void RFAPP_DeviceDataInfoInit (void)	{
    UINT32	field = 0;

	//	clear device data field parameters
	memset ((UINT8*)&RFAPPDevDataField, 0, sizeof(RFAPPDevDataField));
	
	for (field=0; field<=RF_DEVICE_INFO_MAX_DATA_FIELD_INDEX; field++)	{
		//	get latest data from the host application
		RFIntHostApp.pF_GetDeviceDataField (field, RFAPPDevDataField.field[field].data, &RFAPPDevDataField.field[field].len);		
		//	calculate and store crc 
		RFAPPDevDataField.field[field].crc = 0;
		if (RFAPPDevDataField.field[field].len)	{
			RFIntHostApp.pF_CalculateBlockCRC16 (RFAPPDevDataField.field[field].len, &RFAPPDevDataField.field[field].crc, RFAPPDevDataField.field[field].data);
		}
	}	
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief		device information data string request received from gateway   		
 *	
 * \param       UINT8 field
 *
 * \note
 *
 * \warning
 *
 */
INT32	RFAPP_PrepareAndPostDeviceDataInfoStr (UINT8 field)	{	
	UINT32	fd, len;
	INT32	res = SUCCESS;
	INT8    buf[200];

	if			(field <= RF_DEVICE_INFO_MAX_DATA_FIELD_INDEX)	{
		//	post packet to the stack
		RFTransmit_BuildDeviceFieldDataReplyPacketAndPostToPipeByIndex(DEVICE_DESCRIPTIVE_TABLE_INDEX,
											field,
											RFAPPDevDataField.field[field].crc, 
											(UINT8*)RFAPPDevDataField.field[field].data, 
											(UINT8)RFAPPDevDataField.field[field].len);
	}	else if (field == RF_DEVICE_INFO_CRC_FIELD_INDEX)	{
		//	prepare and post the device info crc string	
		for (fd=0, len=0; fd<=RF_DEVICE_INFO_MAX_DATA_FIELD_INDEX; fd++)	{
			//	prepare the crc string
			memcpy((INT8*)&buf[fd*2], (INT8*)&RFAPPDevDataField.field[fd].crc, 2);
			len += 2;
		}
		//	post packet to the stack
		RFTransmit_BuildDeviceFieldDataReplyPacketAndPostToPipeByIndex(DEVICE_DESCRIPTIVE_TABLE_INDEX,
																	   RF_DEVICE_INFO_CRC_FIELD_INDEX, 
																	   0, (UINT8*)buf, len);		
	}	else	{
		//	out of range
		res = FAILURE;
	}

	return (res);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		device information data has changed, update image and report to gateway  		
 *				this function gets called from the host when the device data info string changes
 *	
 * \param       UINT8 field, INT8 *pbuf, UINT16 len
 *
 * \note
 *
 * \warning
 *
 */
INT32	RFAPP_DeviceDataInfoStrChange (UINT8 field, INT8 *pbuf, UINT16 len)	{
	INT32   res = FAILURE;

	if ((field <= RF_DEVICE_INFO_MAX_DATA_FIELD_INDEX) && (len < RF_DEVICE_DATA_FIELD_MAX_SIZE))	{	 
		memset (RFAPPDevDataField.field[field].data, 0, sizeof(RFAPPDevDataField.field[field].data));
		RFAPPDevDataField.field[field].len = len;
		//	update image
		memcpy (RFAPPDevDataField.field[field].data, pbuf, RFAPPDevDataField.field[field].len);
		//	calculate and store crc 
		RFAPPDevDataField.field[field].crc = 0;
		if (RFAPPDevDataField.field[field].len)	{
			RFIntHostApp.pF_CalculateBlockCRC16 (RFAPPDevDataField.field[field].len, &RFAPPDevDataField.field[field].crc, RFAPPDevDataField.field[field].data);
		}			
		//	post packet to rf stack
		RFAPP_PrepareAndPostDeviceDataInfoStr (field);		
		res = SUCCESS;
	}	
	
	return (res);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief		this function process's device's info string requests from the gateway  
 *				the field data is posted to the device
 *	
 * \param       UINT8 index, UINT8 rfid, TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY &pd, UINT16 len
 *
 * \note
 *
 * \warning
 *
 */
void	RFAPP_ProcessDeviceDataInfoStrRequest (UINT8 index, UINT8 rfid, TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY *pd, UINT16 len)	{
	RFAPP_PrepareAndPostDeviceDataInfoStr (pd->hdr.field);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief		this function process's the device info strings CRC's, if the data CRC does not match a request is issued to the
 *				device to send the updated string   		
 *	
 * \param       UINT8 index, UINT8 rfid, TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY &pd, UINT16 len
 *
 * \note
 *
 * \warning
 *
 */
void	RFAPP_ProcessDeviceDataInfoStrReply (UINT8 index, UINT8 rfid, TYPE_RF_DEVICE_FIELD_PACKET_DATA_REPLY *pd, UINT16 len)	{
	INT8	buf[RF_DEVICE_DATA_FIELD_MAX_SIZE+10];
	UINT32	field = 0;
	UINT16	l, crc;

	if (pd->hdr.field <= RF_DEVICE_INFO_MAX_DATA_FIELD_INDEX)	{
		//	save received data and crc in the descriptor table
		//HPRF_NTOHS(pd->hdr.crc);
		RFDesc_SetSlaveDeviceInfoField (index, pd->hdr.field, pd->data, len-sizeof(pd->hdr), pd->hdr.crc);
	}	else if (pd->hdr.field == RF_DEVICE_INFO_CRC_FIELD_INDEX)	{	
		//	check if the device info string crc matches local image, otherwise request that string value the string value has changed
		for (field=0; field<=RF_DEVICE_INFO_MAX_DATA_FIELD_INDEX; field++)	{
			RFDesc_GetSlaveDeviceInfoField (index, field, buf, &l, &crc);	
			if (memcmp((INT8*)&pd->data[field*2], (INT8*)&crc, 2))	{
				//	request update for this string from the device
				RFTransmit_BuildDeviceFieldDataRequestPacketAndPostToPipeByIndex(index, field);		
			}
		}
	}	else	{
		//	invalid info field
		RFIntHostApp.pF_ErrorPrintf ("ERROR: Invalid field info received from rfid %02x, field %02x", rfid, pd->hdr.field);
	}
}

/**
 * \author      Hazrat Shah	
 *
 * \date        08/01/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function handles the rf discover mode timeout	 
 *	
 * \param       void
 *
 * \note
 *
 * \warning
 *
 */
void	RFAPP_RFNetworkDiscoverHandlerTimer (void)	{	
	UINT8	b = 0;

	if (RFDiscover.state != RFDISCOVER_IDLE_STATE)	{
		if (RFDiscover.timer)	{	
			if (--RFDiscover.timer == 0)	{
				RFAPP_RFNetworkDiscoverHandler (RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);			
			}	
		}	
	}	
}
#ifndef	RF_NETWORK_MASTER_DEVICE
/**
 * \author      John Cheng	
 *
 * \date        12/23/2008
 *
 * \return      void
 *
 * \retval      none
 *
 * \brief		this function handles sending beacon requrest after power up  
 *	
 * \param       void
 *
 * \note
 *
 * \warning
 *
 */
void    RFAPP_ScanGatewayHandler(void)
{
    if( g_roamingEnabled && powerUpScan)
	{
		if(powerUpScan == POWER_UP_SEND_REQ)
		{
			UINT8 index = RFDesc_GetNetworkServicesDescriptiveTableIndex();
			RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);
			RFTransmit_BuildRFBeaconReqPacketAndPostToPipeByIndex (index,RF_DISCOVER_GATEWAYS,TRUE);
			powerUpScan = POWER_UP_CHECK_ROAMING_TABLE;
		}
		else if(powerUpScan == POWER_UP_CHECK_ROAMING_TABLE )
		{
			if(!RFReceive_IsGatewayConnected())
			{
                int i;
				TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;

				for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)	
				{
					if ( (pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) && \
						 (pdst->LQ.linkqualityAtLocalDevice != 0xFF)     && \
						 (pdst->LQ.linkqualityAtLocalDevice != 0) )
					{
						if( memcmp((UINT8*)pdst->IDInfo.uid,g_pRFDeviceDescriptiveTable[0].IDInfo.uid, UID_MAX_SIZE) == 0 ) 
						{
							//printf("[%d]same uid\r\n",OS_RetrieveClock());
						}
						else
						{
							//printf("[%d]try to connect to %d\r\n",OS_RetrieveClock(),i);
							RFAPP_LinkRequestHandlerStop();
							RFIntHostApp.pF_StoreGatewayUID_PANID(pdst->IDInfo.uid, pdst->IDInfo.panid,GATEWAY_ADDR_MODE);
							RFIntRFApp.pF_LookupSettingsAndConnectDeviceToNetwork();    
						}
						break;
					}				
				}   
			}
			
			powerUpScan = POWER_UP_SKIP;
		}
	}
}
#endif
/**
 * \author      Hazrat Shah	
 *
 * \date        08/01/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function adds the device to the list  
 *	
 * \param       TYPE_RF_DISCOVERY_REPLY_PACKET_DATA *padreply
 *
 * \note
 *
 * \warning
 *
 */
INT32	RFAPP_AddToDiscoveredDevicesList(TYPE_RF_DISCOVERY_REPLY_PACKET_DATA *preply, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 channel, UINT8 linkquality, UINT8 *pdevindex)	{
	BOOL	found = FALSE;
	UINT32  i;
	INT32	res = FAILURE;

	if (RFDiscover.cnt < MAX_AUTO_DISCOVERED_DEVICES)	{
		res = SUCCESS;
		//	check if device exist in the list	
		for (i=0, found = FALSE; i<RFDiscover.cnt; i++)	{			
			if ( (!memcmp((UINT8*)&RFDiscover.reply[i].uid, puid, sizeof(UID_MAX_SIZE))) && 
				 (!memcmp((UINT8*)&RFDiscover.reply[i].disc.name, (UINT8*)preply->name, strlen((INT8*)preply->name))) && 
				 (RFDiscover.reply[i].addrmode == addrmode)	&&
				 (RFDiscover.reply[i].panid == panid)	&&
				 (RFDiscover.reply[i].disc.tsid == preply->tsid)    &&
				 (RFDiscover.reply[i].disc.rfid == preply->rfid)	&&
				 (!memcmp((UINT8*)&RFDiscover.reply[i].disc.gatewayuid, (UINT8*)preply->gatewayuid, sizeof(preply->gatewayuid))) && 
				 (RFDiscover.reply[i].disc.gatewaypanid == preply->gatewaypanid)  &&
				 (RFDiscover.reply[i].channel == channel))	{
				memcpy((UINT8*)&RFDiscover.reply[i].disc, (UINT8*)preply, sizeof(TYPE_RF_DISCOVERY_REPLY_PACKET_DATA));
	 			RFDiscover.reply[i].linkqualityatlocaldevice = linkquality;
				found = TRUE;
				*pdevindex = i;
				break;
			}
		}
		if (!found)	{
			//	add device to the list
			memcpy((UINT8*)&RFDiscover.reply[RFDiscover.cnt].disc, (UINT8*)preply, sizeof(TYPE_RF_DISCOVERY_REPLY_PACKET_DATA));
			memcpy((UINT8*)&RFDiscover.reply[RFDiscover.cnt].uid, puid, UID_MAX_SIZE);
			RFDiscover.reply[RFDiscover.cnt].panid = panid;
			RFDiscover.reply[RFDiscover.cnt].addrmode = addrmode;
			RFDiscover.reply[RFDiscover.cnt].channel = channel;
			RFDiscover.reply[RFDiscover.cnt].linkqualityatlocaldevice = linkquality;
			*pdevindex = RFDiscover.cnt;
			RFDiscover.cnt++;			
		}
	}	else	{
		//	list is full (show error)				
	}

	return (res);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function prepare and post the rf discover request packet 
 *	
 * \param       UINT32 flags, BOOL broadcast, UINT8 repeats
 *
 * \note
 *
 * \warning
 *
 */
INT32	RFAPP_PrepareAndSubmitRFDiscoverReqPacket (UINT32 flags, BOOL broadcast, UINT8 repeats)	{
    UINT8 index = RFDesc_GetNetworkServicesDescriptiveTableIndex();
	
	RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);
	RFTransmit_BuildRFNetDiscoverRequestPacketAndPostToPipeByIndex (index, flags, broadcast, repeats);

	return (SUCCESS);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function starts the rf network scan process 
 *	
 * \param       TYPE_RF_DISCOVER_REQUEST *adreq
 *
 * \note
 *
 * \warning
 *
 */
INT32	RFAPP_RFNetworkDiscoverHandler (UINT32 trigsrc, UINT32 par2, UINT8 *pBuf, UINT16 len)	{
	INT32	res = SUCCESS;
	TYPE_RFAPP_DATA_BUF *prfapp = (TYPE_RFAPP_DATA_BUF*)pBuf;
	TYPE_RF_NETWORK_SERIVCES_PACKET_DATA *pns = (TYPE_RF_NETWORK_SERIVCES_PACKET_DATA*)&prfapp->data;
	UINT8	b, tblindex;
//	UINT32	i;
	BOOL	isTraceEnabled;
	BOOL	done = FALSE;

	RFIntHostApp.pF_IsRFTxDebugTraceEnabled(&isTraceEnabled);

	switch (RFDiscover.state)	{
		case RFDISCOVER_IDLE_STATE:
			//	not running			
			break;
		case RFDISCOVER_INIT_STATE:
			RFDiscover.scannedchannel = RFDiscover.req.firstchannel;
			//	clear device count
			RFDiscover.cnt = 0;
			RFDiscover.inchrepeat = 0;
		#ifndef	RF_NETWORK_MASTER_DEVICE
			//	disable link handler
			//	RFAPP_LinkRequestHandlerStop();
		#endif
			//if (isTraceEnabled)	{
			//	RFIntHostApp.pF_ConsolePrintf("\r\n\r\nscanning channel(s) %d", RFDiscover.scannedchannel);
			//}	
			// continue to the next state 
		case RFDISCOVER_REQUEST_STATE:
			//	switch channel
			RFTransmit_BuildChannelSelectionFrameAndPostToPipe (RFDiscover.scannedchannel);
			//	wait for channel to switch
			/*TODO: milliseconds? */
			HwDelayMsec(4);
			//	define entry in the pending packet req queue so, the reply gets routed to this function	
			RFAPP_AddEntryofGivenRFPendingPacketReq ((void*)RFAPP_RFNetworkDiscoverHandler, RF_PACKET_DISCOVER_NETDEV_REQUEST, 0, RF_PACKET_DISCOVER_NETDEV_REPLY);
			//	Post discover request packet
			RFAPP_PrepareAndSubmitRFDiscoverReqPacket (RF_DISCOVER_GATEWAYS, TRUE, 0);	
			//	Program timer to wait for link reply
			RFDiscover.timer = RFDISCOVER_REPLY_DELAY;
			RFDiscover.state = RFDISCOVER_REPLY_STATE;		
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf("\tscanning channel %d\r\n", RFDiscover.scannedchannel);
			}
			break;
		
		case RFDISCOVER_REPLY_STATE:
			if (trigsrc == RFREQPKT_TRIGSRC_TIMER_EXPIRED)	{
				if (++RFDiscover.inchrepeat >= RFDISCOVER_MAX_REPEATS)	{
					//if ((RFDiscover.scannedchannel == RFDiscover.req.lastchannel) && (RFDiscover.req.repeats==0)) {
					if (RFDiscover.scannedchannel == RFDiscover.req.lastchannel) {
						done = TRUE;
					}	else	{
						//RFDiscover.req.repeats--;					
						RFDiscover.inchrepeat = 0;
						RFDiscover.scannedchannel = RFAPP_GetNextChannel (RFDiscover.scannedchannel);
						//	continue with the next channel 
						RFDiscover.state = RFDISCOVER_REQUEST_STATE;
						RFAPP_RFNetworkDiscoverHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);			
					}
				}	else	{
					//	try the same channel again
					RFDiscover.state = RFDISCOVER_REQUEST_STATE;
					RFAPP_RFNetworkDiscoverHandler(RFREQPKT_TRIGSRC_TIMER_EXPIRED, 0, &b, 0);			
				}
			}	else	{
				//	device found, add to the list 
				if (SUCCESS == RFAPP_AddToDiscoveredDevicesList((TYPE_RF_DISCOVERY_REPLY_PACKET_DATA*) &pns->pkt.adreply, 
													g_pRFDeviceDescriptiveTable[prfapp->inthdr.index].IDInfo.uid,
													g_pRFDeviceDescriptiveTable[prfapp->inthdr.index].IDInfo.panid,
													g_pRFDeviceDescriptiveTable[prfapp->inthdr.index].IDInfo.addrmode,
													RFDiscover.scannedchannel,
													g_pRFDeviceDescriptiveTable[prfapp->inthdr.index].LQ.linkqualityAtLocalDevice, &tblindex))	{
					//	make call-back
					if (RFDiscover.req.pcbf)	{
						RFDiscover.req.pcbf (&RFDiscover.reply[tblindex] ,RFDiscover.req.detailview);
					}				
				}				
				//	wait for more responses
				//if (RFDiscover.timer <= 2)	{	RFDiscover.timer += 1;	}
				if (!RFDiscover.req.waitforto && !RFDiscover.inchrepeat && (RFDiscover.scannedchannel == RFDiscover.req.lastchannel))	{
					done = TRUE;
				}
			}
			
			if (done)	{	
				//	discovery is done, restore normal operation 
				RFDiscover.state = RFDISCOVER_IDLE_STATE;
				RFDiscover.timer = 0;
				//	restore channel
				RFReceive_SetLocalDeviceChannel(RFReceive_GetLocalDeviceChannel());
				RFAPP_DeleteEntryofGivenRFPendingPacketReq ((void*) RFAPP_RFNetworkDiscoverHandler, RF_PACKET_DISCOVER_NETDEV_REQUEST, 0);
	#ifndef	RF_NETWORK_MASTER_DEVICE
				//	establish link with gw
				//	RFAPP_LinkRequestHandlerStart(0);
	#endif
			}			
			break;
		default:
			res = FAILURE;
			break;
	}

	return (res);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        07/09/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function starts the rf network scan process 
 *	
 * \param       TYPE_RF_DISCOVER_REQUEST *preq
 *
 * \note
 *
 * \warning
 *
 */
INT32	RFAPP_RFNetworkDiscover(TYPE_RF_DISCOVER_REQUEST *preq)	{
	INT32	res = FAILURE;
	UINT32	i;
	UINT8	b;

	//if	(RFAPP_IsNetworkServiceEnabled (preq->service))	{
		if (!preq->scan)	{
			//	display current list
			if (preq->pcbf)	{
				if (RFDiscover.cnt)	{
					for (i=0; i<RFDiscover.cnt; i++)	{
						preq->pcbf(&RFDiscover.reply[i], preq->detailview);
					}
				}	else	{				
					// list empty
				}
			}
		}	else if (RFDiscover.state == RFDISCOVER_IDLE_STATE)	{
			memcpy((UINT8*)&RFDiscover.req, (UINT8*)preq, sizeof(TYPE_RF_DISCOVER_REQUEST));
			RFDiscover.state = RFDISCOVER_INIT_STATE;		
			RFDiscover.timer = 0;	
			RFAPP_RFNetworkDiscoverHandler (0, 0, &b, 0);
		}
	//}

	return (res);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        08/04/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		This function computes network statistics e.g. overall network congestion and  
 *				network utilization
 * \param      
 *
 * \note
 *
 * \warning
 *
 */
void	RFAPP_NetworkStatistics (void)	{
	UINT32	cong, util, txpkts, rxpkts, txtdly, turnarounddly, i, devcnt;
	#define	RF_PKT_EST_FACTORIAL	(65535/90)

	//	overall network congestion 
	//	congestion equation derived/optimized based on this net characteristics 
	txpkts = (Rfdiags_getLowLevelTxPacketCnt() - NetBwSt.NumTxPktsCong);
	rxpkts = (Rfdiags_getLowLevelRecvCnt() - NetBwSt.NumRxPktsCong);
	for (i=0, turnarounddly=0, devcnt=0; i<g_MaxSupportedRFDevices; i++)	{
		if ((g_pRFDeviceDescriptiveTable[i].linkstate == RF_DEVICE_LINK_STATE_ACTIVE) &&
			(g_pRFDeviceDescriptiveTable[i].COMLog.txpacketcount-g_pRFDeviceDescriptiveTable[i].COMLog.txpacketretrycount))	{
			turnarounddly += g_pRFDeviceDescriptiveTable[i].COMLog.txpacketaccturnaroundtime/(g_pRFDeviceDescriptiveTable[i].COMLog.txpacketcount-g_pRFDeviceDescriptiveTable[i].COMLog.txpacketretrycount);
			devcnt++;
		}
	}
	if (devcnt) {
		if (turnarounddly > NetBwSt.TxTurnArounddlyCong)	{
			txtdly = (turnarounddly - NetBwSt.TxTurnArounddlyCong)/devcnt;
		}	else	{
			txtdly = (NetBwSt.TxTurnArounddlyCong - turnarounddly)/devcnt;
		}
	}	else	{
		txtdly = 0; 
	}
	NetBwSt.TxTurnArounddlyCong = turnarounddly;
	NetBwSt.NumTxPktsCong = Rfdiags_getLowLevelTxPacketCnt();
	NetBwSt.NumRxPktsCong = Rfdiags_getLowLevelRecvCnt();
	cong = (rxpkts + txpkts + txtdly) * RF_PKT_EST_FACTORIAL;
	if (cong > 65535)	{	cong = 65535;	}
	Rfdiags_Congestion = cong;

	//RFIntHostApp.pF_ConsolePrintf ("Cong: txpkt %u rxpkt %u cong %3.2f\r\n", rxpkts, rxpkts, (float) (cong/655));

	//	net utilization by this device
	rxpkts = (Rfdiags_getLowLevelRecvCnt()-Rfdiags_getLowLevelInvalidRxPacketCnt()) - NetBwSt.NumrxPktsUtil;
	txpkts = Rfdiags_getLowLevelTxPacketCnt() - NetBwSt.NumtxPktsUtil;
	NetBwSt.NumrxPktsUtil  = (Rfdiags_getLowLevelRecvCnt()-Rfdiags_getLowLevelInvalidRxPacketCnt());
	NetBwSt.NumtxPktsUtil  = Rfdiags_getLowLevelTxPacketCnt();
	util   = (txpkts + rxpkts) * RF_PKT_EST_FACTORIAL;
	if (util > 65535)	{	util = 65535;	}
	Rfdiags_Utilization = util;

	//RFIntHostApp.pF_ConsolePrintf ("Util: txpkt %u rxpkt %u utl %3.2f\r\n", rxpkts, rxpkts, (float) (util/655));
}

/**
 *
 * \date        03-21-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       This function returns a random number	     
 * 
 * \detail      
 *
 * \param		void
 *
 * \return      UINT16 	
 *
 * \retval      
 *
 */
UINT16 RFAPP_GetRandomNum (void)	{
	UINT8  uid[UID_MAX_SIZE];
	RFReceive_GetLocalDeviceUID (uid);

	return (RFReceive_GetTimerValue() ^ ((UINT16)uid[0]) ^ ((UINT16)uid[2]));	
}


/**
 *
 * \date        08-11-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Pepares PassTo command	     
 * 
 * \detail      
 *
 * \param		UINT8 rfid
 *
 * \return      INT32 	
 *
 * \retval      
 *
 */
INT32	RFAPP_PrepareAndSubmitPassToEnterCmd (UINT8 rfid)	{
	INT32	res = SUCCESS;
	UINT8	buf[50];
	INT32(*pcbf)(UINT32 par, UINT32 status) = NULL;

	buf[0] = rfid;
	buf[1] = 3;
	buf[2] = 0x12; //CNET_MULTISERIAL;
	buf[3] = 0x80;
	buf[4] = 0x00;

	res = RFTransmit_BuildCresnetPacketAndPostToPipeByRFID (rfid, buf, buf[1]+2, pcbf, 0, 0);		

	return (res);
}

/**
 *
 * \date        08-11-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Pepares PassTo exit command	     
 * 
 * \detail      
 *
 * \param		UINT8 rfid
 *
 * \return      INT32 	
 *
 * \retval      
 *
 */
INT32	RFAPP_PrepareAndSubmitPassToExitCmd (UINT8 rfid)	{
	INT32	res = SUCCESS;
	UINT8	buf[50];
	INT32(*pcbf)(UINT32 par, UINT32 status) = NULL;

	buf[0] = rfid;
	buf[1] = 3;
	buf[2] = 0x12;	//CNET_MULTISERIAL;
	buf[3] = 0x80;
	buf[4] = 0x01;

	res = RFTransmit_BuildCresnetPacketAndPostToPipeByRFID (rfid, buf, buf[1]+2, pcbf, 0, 0);		

	return (res);
}

/**
 *
 * \date        08-11-2007
 *
 * \author      Hazrat Shah	
 *
 * \brief       Submit data to device in PassTo mode 	     
 * 
 * \detail      
 *
 * \param		UINT8 rfid, INT8 *pbuf, UINT8 len
 *
 * \return      INT32 	
 *
 * \retval      
 *
 */
INT32 RFAPP_PrepareAndSubmitPassToCmd (UINT8 rfid, INT8 *pbuf, UINT8 len)	{
	INT32	res = SUCCESS;
	UINT8	buf[300];
	INT32(*pcbf)(UINT32 par, UINT32 status) = NULL;

	buf[0] = rfid;
	buf[1] = 2+strlen(pbuf);
	buf[2] = 0x12;	//CNET_MULTISERIAL;
	buf[3] = 0x00;
	memcpy(&buf[4], pbuf, len);

	res = RFTransmit_BuildCresnetPacketAndPostToPipeByRFID (rfid, buf, buf[1]+2, pcbf, 0, 0);	

	return (res);
}

/**
 * \author      Hazrat Shah	
 *
 * \date        08/12/2007
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function prepare and post the remote setup cfg packet 
 *	
 * \param       UINT32 flags, UINT8 *puid, UINT16 panid, UINT8 addrmode
 *
 * \note
 *
 * \warning
 *
 */
 INT32	RFAPP_PrepareAndSubmitRFRemoteCfgPacket (UINT32 flags, UINT8 *pdevuid, UINT16 devpanid, UINT8 devaddrmode, UINT8 *pgwuid, UINT16 gwpanid, UINT8 gwaddrmode) {
    UINT8 index = RFDesc_GetNetworkServicesDescriptiveTableIndex();
	
	RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);
	RFTransmit_BuildRFNetRemoteSetupPacketAndPostToPipeByIndex (index, flags, pdevuid, devpanid, devaddrmode,  pgwuid, gwpanid, gwaddrmode);

	return (SUCCESS);
}
#ifdef	RF_NETWORK_MASTER_DEVICE
/**
 * \author      John Cheng	
 *
 * \date        12/4/2008
 *
 * \return      INT32
 *
 * \retval      SUCCESS/FAILURE
 *
 * \brief		this function prepare and post the beacon packet 
 *	
 * \param       none
 *
 * \note
 *
 * \warning
 *
 */
void RFAPP_PrepareAndSubmitRFBeaconPacket (void) 
{
    UINT8 index = RFDesc_GetNetworkServicesDescriptiveTableIndex();
	TYPE_RFSTACK_CB cb;
	memset ((UINT8*)&cb, 0, sizeof(cb));

	RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);
	RFTransmit_BuildRFNetDiscoverBeaconPacketAndPostToPipeByIndex (index, 0, TRUE, FALSE, &cb);
}
#endif
 /**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function handles the control system connections status to be reported to the 
 *				slave devices
 *
 * \detail			 
 *				
 *
 *						
 * \date        09/27/2007
 *					
 * \param	    UINT32 par, UINT32 newstate
 *
 * \return		None	  					
 *
 * \retval		None
 */
void RFAPP_ControlSystemStatusHandler (UINT32 par, UINT32 newstate)	{
	
	RFAppNetMon.NetMonTimer = 0;

	switch (newstate)	{
		case RFAPP_CIP_ONLINE:
			if (RFAppNetMon.CresnetConnected && !RFAppNetMon.CIPOnLine)	{
				RFAppNetMon.ControlSystemConn = FALSE;				
				RFAppNetMon.NetMonTimer = 2; 
			}	else	{
				RFAppNetMon.ControlSystemConn = TRUE;				
			}
			RFAppNetMon.NetMonState = newstate;	
			RFAppNetMon.CIPOnLine = TRUE;
			RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST);	
			break;
		case RFAPP_CIP_OFFLINE:
			if (RFAppNetMon.CresnetConnected)	{
				RFAppNetMon.ControlSystemConn = FALSE;					
				RFAppNetMon.NetMonTimer = 2; 	
				RFAppNetMon.NetMonState = RFAPP_CRESNET_CONN;
			}	else	{
				RFAppNetMon.ControlSystemConn = FALSE;
				RFAppNetMon.NetMonState = newstate;
			}
			RFAppNetMon.CIPOnLine = FALSE;
			RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST);	
			break;
		case RFAPP_CRESNET_CONN:
			RFAppNetMon.ControlSystemConn = TRUE; 
			RFAppNetMon.CresnetConnected = TRUE;	
			RFAppNetMon.NetMonState = RFAPP_CRESNET_CONN;
			RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST);	
			break;		
		case RFAPP_CRESNET_DISCONN:
			if (!RFAppNetMon.CIPOnLine)	{
				RFAppNetMon.ControlSystemConn = FALSE;
			}
			RFAppNetMon.NetMonState = RFAPP_CRESNET_DISCONN;
			RFAppNetMon.CresnetConnected = FALSE;
			RFTransmit_BuildCSStatusPacketAndPostToPipeByRFID (RFS_CNET_BROADCAST);	
			break;
	}				
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function handles the control system connections status handler timeout 
 *
 * \detail			 
 *				
 *
 *						
 * \date        09/27/2007
 *					
 * \param	    None
 *
 * \return		None	  					
 *
 * \retval		None
 */
void RFAPP_ControlSystemStatusHandlerTimer (void)	{
	if (RFAppNetMon.NetMonTimer)	{
		if (--RFAppNetMon.NetMonTimer == 0)	{
			RFAPP_ControlSystemStatusHandler (0, RFAppNetMon.NetMonState);
		}
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns TRUE if control system is connected and online 
 *
 * \detail			 
 *				
 *
 *						
 * \date        09/27/2007
 *					
 * \param	    None
 *
 * \return		BOOL	  					
 *
 * \retval		TRUE/FALSE
 */
BOOL RFAPP_IsControlSystemOnLine (void)	{
	return (RFAppNetMon.ControlSystemConn);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns TRUE if cresent is connected and polling  
 *
 *
 * \detail			 
 *				
 *
 *						
 * \date        09/27/2007
 *					
 * \param	    None
 *
 * \return		BOOL	  					
 *
 * \retval		TRUE/FALSE
 */
BOOL RFAPP_IsCresenetOnLine (void)	{
	return (RFAppNetMon.CresnetConnected);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns TRUE if CIP is on line	  
 *
 * \detail			 
 *
 *						
 * \date        09/27/2007
 *					
 * \param	    None
 *
 * \return		BOOL	  					
 *
 * \retval		TRUE/FALSE
 */
BOOL RFAPP_IsCIPOnLine (void)	{
	return (RFAppNetMon.CIPOnLine);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function enables/disables the network services		  
 *
 * \detail			 
 *						
 * \date        10/02/2007
 *					
 * \param	    UINT8 serviceID, UINT8 enable
 *
 * \return		INT32		  					
 *
 * \retval		TRUE/FALSE
 */
INT32	RFAPP_EnableDisableNetworkService (UINT8 serviceID, UINT8 enable)	{
	INT32 res = SUCCESS;

	switch (serviceID)	{
		case RF_PACKET_DISCOVER_NETDEV_REQUEST:
			if (enable)	{
				NetworkServiceEnableFlags |= (1<<RF_PACKET_DISCOVER_NETDEV_REQUEST);
			}	else	{
				NetworkServiceEnableFlags &= ~(1<<RF_PACKET_DISCOVER_NETDEV_REQUEST);
			}
			break;
		case RF_PACKET_DISCOVER_NETDEV_REPLY:
			if (enable)	{
				NetworkServiceEnableFlags |= (1<<RF_PACKET_DISCOVER_NETDEV_REPLY);
			}	else	{
				NetworkServiceEnableFlags &= ~(1<<RF_PACKET_DISCOVER_NETDEV_REPLY);
			}
			break;
		case RF_PACKET_REMOTE_SETUP:
			if (enable)	{
				NetworkServiceEnableFlags |= (1<<RF_PACKET_REMOTE_SETUP);
			}	else	{
				NetworkServiceEnableFlags &= ~(1<<RF_PACKET_REMOTE_SETUP);
			}
			break;
		default:
			res = FAILURE;
			break;
	}

	return (res);
}


/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks if the network service is enables 		  
 *
 * \detail			 
 *						
 * \date        10/02/2007
 *					
 * \param	    UINT8 serviceID
 *
 * \return		BOOL			  					
 *
 * \retval		TRUE/FALSE
 */
BOOL	RFAPP_IsNetworkServiceEnabled (UINT8 serviceID)	{
	BOOL enabled = FALSE;

	switch (serviceID)	{
		case RF_PACKET_DISCOVER_NETDEV_REQUEST:
			enabled = (NetworkServiceEnableFlags & (1<<RF_PACKET_DISCOVER_NETDEV_REQUEST)?TRUE:FALSE);
			break;
		case RF_PACKET_DISCOVER_NETDEV_REPLY:
			enabled = (NetworkServiceEnableFlags & (1<<RF_PACKET_DISCOVER_NETDEV_REPLY)?TRUE:FALSE);
			break;
		case RF_PACKET_REMOTE_SETUP:
			enabled = (NetworkServiceEnableFlags & (1<<RF_PACKET_REMOTE_SETUP)?TRUE:FALSE);
			break;
	}

	return (enabled);
}

/**********************************************************************************************/
/*               The following section is for RF roaming                                      */
/**********************************************************************************************/
/**
 * \author      John Cheng
 * \brief       update seleted gateway slot UID string on setup page
 * \date        12/1/2008 11:37AM
 * \param       none
 *
 * \return      none
 * \retval      none
 */
void RFRM_Init(void)
{
#if RFRM_READY
    gPowerBoost = param.rfparam.powerBoost;

    g_roamingEnabled = param.rfrmparam.roaming;
    g_testRoamingBufSize = param.rfrmparam.MV_averageFilterSize;
    if (g_testRoamingBufSize > GW_RM_MAX_MV_AV_FILTER_SZ
            || g_testRoamingBufSize <= 0)
    {
        g_testRoamingBufSize = GW_RM_DEFALT_MV_AV_FILTER_SZ;
        param.rfrmparam.MV_averageFilterSize = g_testRoamingBufSize;
        TriggerParameterSave();
    }

    dig_set_state(RF_ROAMING_DISABLE, !g_roamingEnabled);
    dig_set_state(RF_ROAMING_ENABLE, g_roamingEnabled);
    dig_set_state(RF_SETUP_GW_1, g_roamingEnabled);

    //record_analog_change(RF_ACQUIRE_ROAMING_GW_SLOT, 1);//from 1 ~ 8
    g_acquireSlot = 1;
    dig_set_state(RF_SETUP_GW_2, 0);
    dig_set_state(RF_SETUP_GW_3, 0);
    dig_set_state(RF_SETUP_GW_4, 0);
    dig_set_state(RF_SETUP_GW_5, 0);
    dig_set_state(RF_SETUP_GW_6, 0);
    dig_set_state(RF_SETUP_GW_7, 0);
    dig_set_state(RF_SETUP_GW_8, 0);

    record_analog_change(RF_ROAMING_GW_CONNECTED_INDEX, 0);
    if (param.rfrmparam.TriggerLPDOnGWPeriod < 1
            || param.rfrmparam.TriggerLPDOnGWPeriod > 5)
        param.rfrmparam.TriggerLPDOnGWPeriod = 1;

    if (param.rfrmparam.roamingAggressLevel >= GW_RM_THRESHOLD_MAX_LEVEL)
        param.rfrmparam.roamingAggressLevel = ROAMING_AGG_DEFAULT;
    gGW_RM_threshold = GW_Aggressiveness[param.rfrmparam.roamingAggressLevel];
    record_analog_change(RF_ROAMING_AGGRESIVE_LEVEL,param.rfrmparam.roamingAggressLevel);
#endif

    RfRmTaskStr.task = OsCreateNamedAdvTaskPri(RFRM_Task, 0, 0, RFAPP_TASK_PRIORITY, "ERRFRMTask");

    if (!RfRmTaskStr.task)
    {
        RFIntHostApp.pF_ErrorPrintf("RFRMAPP: Error create RFRM_Task thread!");
        RFIntHostApp.pF_SystemError(FATAL_RF_INIT);
    }

    if(g_roamingEnabled)
    {
        RFRM_updateRoamingGW_connectedIndex();
#if RFRM_READY
        RF_UpdateSelectedGatewaySlotUID();
#endif
    }

}
/**
 * \author      John Cheng
 * \brief       call back funciton of echo response
 * \detail
 * \date        12/23/2008
 * \param       TYPE_RF_ECHO_REPLY_PACKET *pechoreply
 * \return      INT32
 * \retval      SUCCESS
 */
INT32   RF_RM_EchoResponse (TYPE_RF_ECHO_REPLY_PACKET *pechoreply)
{
    INT8 curConn ;

    if (pechoreply->done)   {
        //NOTE: pF_ConsolePrintf will check debug flag
        {
            RFIntHostApp.pF_ConsolePrintf("\r\n\ttotal tests: %d\r\n", pechoreply->repeatno);
            RFIntHostApp.pF_ConsolePrintf("\tinvalid responses: %d\r\n", pechoreply->invalidresponses);
            RFIntHostApp.pF_ConsolePrintf("\tno reponses: %d\r\n", pechoreply->noresponses);
            RFIntHostApp.pF_ConsolePrintf("\tlink quality average at local device:  %d dBm \r\n", RFReceive_ConvertLinkQualityTodBm(pechoreply->avgrxlqi));
            RFIntHostApp.pF_ConsolePrintf("\tlink quality average at remote device: %d dBm \r\n", RFReceive_ConvertLinkQualityTodBm(pechoreply->avgremotedevicerxlqi));
            RFIntHostApp.pF_ConsolePrintf("\ttransmitted pkts: %d\r\n", pechoreply->txpacketcount);
            RFIntHostApp.pF_ConsolePrintf("\treceived pkts: %d\r\n", pechoreply->rxpacketcount);
            RFIntHostApp.pF_ConsolePrintf("\ttransmitted data bytes: %d\r\n", pechoreply->txpacketpayloadbytecount+pechoreply->rxpacketpayloadbytecount);
        }


        if(RF_RM_findCurrentConnectedByIndex(&curConn) && pechoreply->repeatno > (pechoreply->invalidresponses+pechoreply->noresponses) )
        {
            g_pRFRoamingDescriptiveTable[curConn].linkeactiveestablishedtime = OS_RetrieveClock();
            g_pRFRoamingDescriptiveTable[curConn].RxBuf.data[RF_GW_LQ_BUFFER_SIZE] = pechoreply->localrxlqi;
        }

    }
    return SUCCESS;
}
/**
 * \author      John Cheng
 * \brief       send analog joins to app.
 * \detail
 * \date        3/23/2014
 * \param       UINT16 data
 * \return      none
 * \retval      none
 */
void RF_RM_SendAnalogJoinsToApp (UINT16 data)
{
    char buf[4];
#ifdef NEW_ITEM_OF_CTRL_CALLBK
    buf[0] = HPRF_CTRL_MSG_CTRL_SYS_ANALOG;
#endif
    buf[1] = (char) (data >> 4); //high
    buf[2] = (char) (data & 0xFF); //low
    SendCtrlMsgToApp((void*) buf, sizeof(buf));
}
/**
 * \author      John Cheng
 * \brief       send digital joins to app.
 * \detail
 * \date        3/23/2014
 * \param       UINT16 data
 * \return      none
 * \retval      none
 */
void RF_RM_SendDigitalJoinsToApp (UINT16 data)
{
    char buf[4];
#ifdef NEW_ITEM_OF_CTRL_CALLBK
    buf[0] = HPRF_CTRL_MSG_CTRL_SYS_DIGITAL;
#endif
    buf[1] = (char) (data >> 4); //high
    buf[2] = (char) (data & 0xFF); //low
    SendCtrlMsgToApp((void*) buf, sizeof(buf));
}
/**
 * \author      John Cheng
 * \brief       send digital joins to app.
 * \detail
 * \date        3/23/2014
 * \param       UINT16 data
 * \return      none
 * \retval      none
 */
void RF_RM_SendSerialJoinsToApp (UINT16 data)
{
    char buf[4];
#ifdef NEW_ITEM_OF_CTRL_CALLBK
    buf[0] = HPRF_CTRL_MSG_CTRL_SYS_DIGITAL;
#endif
    buf[1] = (char) (data >> 4); //high
    buf[2] = (char) (data & 0xFF); //low
    SendCtrlMsgToApp((void*) buf, sizeof(buf));
}
/**
 * \author      John Cheng
 * \brief       switch gateway
  * \detail
 * \date        12/23/2008
 * \param       INT8* index
 * \return      void
 * \retval      none
 */
void RF_RM_SwitchGWByIndex(INT8 index)
{
    UINT8  channel = SLAVE_DEVICE_DEFAULT_CHANNEL;
    if( index >= 0 && index < MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES )
    {
        if(g_pRFRoamingDescriptiveTable[index].linkstate != RF_DEVICE_LINK_STATE_EMPTY)
        {
            RFIntHostApp.pF_ConsolePrintf("Switch to gateway %d.\r\n",index+1);

            g_roamingInProgress = true;
            roamingStartTime = OS_RetrieveClock();
            if(roamingStartTime == 0)
                roamingStartTime = 1;//roamingStartTime = 0 means popup page is on

            RFIntRFApp.pF_EnterPowerDownMode();
            RFIntHostApp.pF_GetChannel(&channel, SLAVE_DEVICE_DEFAULT_CHANNEL);
            RFIntRFApp.pF_ConnectDeviceToNetwork(g_pRFRoamingDescriptiveTable[index].IDInfo.uid,\
                                                 g_pRFRoamingDescriptiveTable[index].IDInfo.panid,\
                                                 channel,GATEWAY_ADDR_MODE);

            RFIntHostApp.pF_StoreGatewayUID_PANID(g_pRFRoamingDescriptiveTable[index].IDInfo.uid,\
                                                  g_pRFRoamingDescriptiveTable[index].IDInfo.panid,\
                                                  GATEWAY_ADDR_MODE);
            
#ifdef NEW_ITEM_OF_CTRL_CALLBK
            RF_RM_SendAnalogJoinsToApp(RF_ROAMING_GW_CONNECTED_INDEX, index+1);
#endif
            g_pRFRoamingDescriptiveTable[index].COMLog.rxbufoverflowcount = 0;
        }
        else
        {
            //printf("switch to table %d failed.\r\n",index);
        }
    }
}
//0. if roaming is enabled
//1. when power up, need to wait until at least one gateway has send enough beacon to calculate
//2. stop monitoring while acquireing gateway
//3. after acquireing a gateway, do not monitor until ???
//4. if there is at least one acqured gateway in the table

INT8 RF_RM_monitorLQ(void)
{
    INT8 res = 0;
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = NULL;
    int i = 0;
//0. if roaming is enabled
    if( !g_roamingEnabled )
    {
        return 0;
    }

//2. stop monitoring while acquireing gateway
    if(RFAPP_AcquireStatus())
    {
        return 0;
    }

//4. if there is at least one acqured gateway in the table
    pdst = g_pRFRoamingDescriptiveTable;
    for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)
    {
        if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) )
        {
            res = 1;
            break;
        }
    }

    return res;

}
/**
 *
 * \author      John Cheng
 * \brief       to find max strength of gateways
 * \detail
 * \date        12/23/2008
 * \param       INT8* index, INT8* dBm
 * \return      void
 * \retval      none
 */
void RF_RM_findMaxLQStrengthInRoamingTable(INT8* index, INT8* dBm)
{
    UINT8 savedData = 0xFF;
    int i = 0;
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;

    *index = -1;
    for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)
    {
        if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) && pdst->RxBuf.data[RF_GW_LQ_BUFFER_SIZE])
        {
            if( pdst->RxBuf.data[RF_GW_LQ_BUFFER_SIZE] < savedData )
            {
                *index = i;
                savedData = pdst->RxBuf.data[RF_GW_LQ_BUFFER_SIZE];
            }
        }
    }
    *dBm = RFReceive_ConvertLinkQualityTodBm(savedData);
}
/**
 * \author      John Cheng
 * \brief       send Beacon To gateway
 * \detail
 * \date        12/23/2008
 * \param       none
 * \return      void
 * \retval      none
 */
void RF_RM_sendBeaconToGW(UINT32* lastTime)
{
    #define ONE_MINUTE_TICK 60000
    UINT32 curTime = 0;
    UINT8 index = 0;
    //erRadioSettings.ulER_TriggerLowPriorityDelayOnGW;
    //??NOTE: how to get this flag???
    unsigned long ulTriggerLPDOnGWPeriod = 0;

    //??if(!param.rfrmparam.TriggerLowPriorityDelayOnGW)
    if(!ulTriggerLPDOnGWPeriod)
        return;

    curTime = OS_RetrieveClock();
    if( curTime >= *lastTime )
    {
        if( (curTime - *lastTime) > ONE_MINUTE_TICK*ulTriggerLPDOnGWPeriod)//every x minutes
        {
            index = RFDesc_GetNetworkServicesDescriptiveTableIndex();
            RFTransmit_BuildRFBeaconReqPacketAndPostToPipeByIndex (index,0,TRUE);
            *lastTime = curTime;
        }
    }
    else
    {
        *lastTime = curTime;
    }
}
/**
 * \author      John Cheng
 * \brief       send Beacon To gateway
 * \detail
 * \date        12/23/2008
 * \param       none
 * \return      void
 * \retval      none
 */
static void RF_RM_checkGWConn(void)
{
    INT8 curConnIndex = -1;
    int i = 0;

    if(RF_RM_findCurrentConnectedByIndex(&curConnIndex))
    {
        for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++)
        {
            if ((g_pRFRoamingDescriptiveTable[i].linkstate != RF_DEVICE_LINK_STATE_EMPTY) && curConnIndex != i )
            {
                if(g_pRFRoamingDescriptiveTable[i].COMLog.rxbufoverflowcount > GATEWAY_OVERFLOW_THRE )
                {
                    UINT8 index = RFDesc_GetNetworkServicesDescriptiveTableIndex();
                    memcpy(g_pRFDeviceDescriptiveTable[index].IDInfo.uid, &g_pRFRoamingDescriptiveTable[i].IDInfo.uid, UID_MAX_SIZE);
                    g_pRFRoamingDescriptiveTable[i].txsequence += 1;
                    g_pRFDeviceDescriptiveTable[index].txsequence = g_pRFRoamingDescriptiveTable[i].txsequence;
                    RFTransmit_BuildPowerDownModePacketAndPostToPipeByIndex(index);
                    g_pRFRoamingDescriptiveTable[i].COMLog.rxbufoverflowcount = 0;
                    RFIntHostApp.pF_ConsolePrintf("\r\nKilled gateway %d connection.\r\n",i+1);
                }
            }
        }
    }
}

/**
 *
 * \author      John Cheng
 *
 * \brief       main Beacon Update Task
 *
 *
 * \detail
 *
 * \date        12/23/2008
 *
 * \param       unsigned long argc, void* argv
 *
 * \return      INT32
 *
 * \retval      none
 */
static void RFRM_Task(UINT32 param)
{
    #define TASK_LOOPING_INTERVAL 1000
    #define BLOCK_POPUP_TIMEOUT   10000

    unsigned short lq_norm;
    UINT8   GatewayDevLQI=255;
    UINT32  lastMsgTime = 0;
    UINT32  LQ_av = 0;
    INT32   LQ_index = 0;
    int i,j;

    HwDelayMsec(RF_CONN_CHECK_RATE);//??

    while (TRUE)
    {
//update signal strength
        TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;
        if( g_roamingEnabled )
        {
            for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)
            {
                if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) )
                {
                    INT32 diff = OS_RetrieveClock() - pdst->linkeactiveestablishedtime;
                    if( diff > RF_CONN_CHECK_RATE && diff < RF_BEACON_RATE*2 )
                    {
                        INT8 curConn = -1;
                        if(RF_RM_findCurrentConnectedByIndex(&curConn))
                        {
                            if(curConn == i)
                            {
                                TYPE_RF_ECHO_REQUEST_PACKET echoreq;

                                memset ((UINT8*)&echoreq, 0, sizeof(echoreq));
                                RFIntHostApp.pF_GetRFID(&echoreq.rfid);
                                echoreq.repeats = 0;
                                sprintf((char*)echoreq.buf, "echo this connection.");
                                echoreq.buflen  = strlen((char*)echoreq.buf);
                                echoreq.start   = TRUE;
                                echoreq.pcbf    = RF_RM_EchoResponse;

                                RFIntRFApp.pF_PostEchoPacketRequest(&echoreq);
                            }
                        }
                    }

                    if( diff > RF_BEACON_RATE*2 )//using 5 cecond window
                    {
                        GatewayDevLQI = 0xFF;//will make min. value -95dBm
                    }
                    else
                        GatewayDevLQI = pdst->LQ.linkqualityAtLocalDevice;

                    lq_norm= RFIntRFApp.pF_ConvertLQIToIEEEScaleWord(GatewayDevLQI, LQI_DISP_OFFSET);
                    
#ifdef NEW_ITEM_OF_CTRL_CALLBK
                    RF_RM_SendAnalogJoinsToApp(RF_INTERNAL_WIFI_SIGNAL1_STRENGTH + i, lq_norm);
#endif
                    LQ_av = 0;
                    LQ_index = pdst->RxBuf.index;
                    //using moving average filter,filter size is g_testRoamingBufSize
                    if( GatewayDevLQI != 0xFF )
                    {
                        if( LQ_index >= RF_GW_LQ_BUFFER_SIZE )
                            LQ_index = 0;

                        for (j=0;j<g_testRoamingBufSize;j++)
                        {
                            LQ_index-- ;
                            if( LQ_index < 0 )
                                LQ_index = RF_GW_LQ_BUFFER_SIZE-1;

                            LQ_av += pdst->RxBuf.data[LQ_index];
                        }

                        LQ_av = LQ_av / g_testRoamingBufSize ;
                        pdst->RxBuf.data[RF_GW_LQ_BUFFER_SIZE] = (UINT8)LQ_av;
                    }
                    else
                    {
                        pdst->RxBuf.data[RF_GW_LQ_BUFFER_SIZE] = 0xFF;
                    }
                }
            }
        }

//roaming accroding to the join number or commad.
        if( RF_RM_monitorLQ() && g_GW_switch )//switch to the index from 1 to 8
        {
            if( g_GW_switch > 0 && g_GW_switch <= MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES )
            {
                if( memcmp((UINT8*)g_pRFRoamingDescriptiveTable[g_GW_switch -1].IDInfo.uid,\
                               g_pRFDeviceDescriptiveTable[0].IDInfo.uid, UID_MAX_SIZE) != 0 )
                {
                    //printf("RF_RM_SwitchGWByIndex %d;\r\n",g_GW_switch -1);
                    RF_RM_SwitchGWByIndex(g_GW_switch -1);
                }
            }
            g_GW_switch = 0;//reset the index here
        }

//LQ comparing
        if( RF_RM_monitorLQ() )
        {
            INT8 maxLQIndex = -1;
            INT8 curConnIndex = -1;
            INT8 maxLQdBm = 0;
            RF_RM_findMaxLQStrengthInRoamingTable(&maxLQIndex,&maxLQdBm );
            if(maxLQIndex != -1)
            {
                if(RF_RM_findCurrentConnectedByIndex(&curConnIndex))
                {
                    if( maxLQIndex != curConnIndex )
                    {
                        INT16 diff = g_pRFRoamingDescriptiveTable[curConnIndex].RxBuf.data[RF_GW_LQ_BUFFER_SIZE] - \
                                     g_pRFRoamingDescriptiveTable[maxLQIndex].RxBuf.data[RF_GW_LQ_BUFFER_SIZE];

                        if( diff > gGW_RM_threshold)
                        {
                            RF_RM_SwitchGWByIndex(maxLQIndex);
                        }
                    }
                }
                else//target gateway is not in roaming entry table
                {
                    if(RFReceive_IsGatewayConnected())
                    {
                        INT16 diff = g_pRFDeviceDescriptiveTable[0].LQ.linkqualityAtLocalDevice - \
                                     g_pRFRoamingDescriptiveTable[maxLQIndex].RxBuf.data[RF_GW_LQ_BUFFER_SIZE];

                        if( diff > 0 )
                            RF_RM_SwitchGWByIndex(maxLQIndex);
                    }
                    else
                    {
                        RF_RM_SwitchGWByIndex(maxLQIndex);
                    }
                }
            }
        }

        if( RF_RM_monitorLQ() )
        {
            //to trigger Low Priority Message Post Delay on gateway
            RF_RM_sendBeaconToGW(&lastMsgTime);
            // sometimes when roaming, the disconnected gateway will keep sending message to the panel.
            // In this case RFRecMsg_QualifyDataPacket() funciton will print "RF Descriptor entry table
            // full" error mesage. We increase COMLog.rxbufoverflowcount inside RFRecMsg_QualifyDataPacket()
            // funciton for this event.
            // Here we can use this count to find out this problem, and then to disconnect this gateway.
            RF_RM_checkGWConn();
        }

        if( roamingStartTime &&
            ( (RFIntRFApp.pF_GatewayConnectionStatusQuery()!= true) ||
              (RFIntRFApp.pF_ControlSystemConnectionStatusQuery()!= true)) )
        {
            UINT32 curTime = OS_RetrieveClock();
            if( curTime > roamingStartTime )
            {
                if( (curTime - roamingStartTime) > BLOCK_POPUP_TIMEOUT)//about 10 seconds
                {
                    if(RFIntRFApp.pF_GatewayConnectionStatusQuery()!= true)
                    {
                        RFIntHostApp.pF_GatewayConnectionStatusChange(false);
                    }
                    if(RFIntRFApp.pF_ControlSystemConnectionStatusQuery()!= true)
                    {
                        RFIntHostApp.pF_ControlSystemConnectionStatusChange(false,true);
                    }
                    roamingStartTime =0;
                }
            }
            else
            {
                roamingStartTime = curTime;
                if(!roamingStartTime)
                    roamingStartTime = 1;
            }
        }
        else if(g_roamingInProgress)
        {
            g_roamingInProgress = false;
            roamingStartTime = 0;
        }

        HwDelayMsec(TASK_LOOPING_INTERVAL);
    }
}


/**
 * \author      John Cheng
 * \brief       to find which one is currently connected
 * \detail
 * \date        12/23/2008
 * \param       INT8* index
 * \return      INT8
 * \retval      1 = found, 0 = no
 */
INT8 RF_RM_findCurrentConnectedByIndex(INT8* index)
{
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;
    int i = 0;

    for (i = 0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)
    {
        if( memcmp((UINT8*)g_pRFRoamingDescriptiveTable[i].IDInfo.uid,\
                               g_pRFDeviceDescriptiveTable[0].IDInfo.uid, UID_MAX_SIZE) == 0 )
        {
            *index = i;
            return 1;
        }
    }

    return 0;
}
/**
 * \author      John Cheng
 * \brief       to update the analog join
 * \detail
 * \date        12/23/2008
 * \param       none
 * \return      void
 * \retval      none
 */
void   RFRM_updateRoamingGW_connectedIndex(void)
{
    BOOL    gatewayuidvalid;
    UINT8   gatewayuid[UID_MAX_SIZE];
    UINT16  gatewaypanid;
    UINT8   gatewayaddrmode;
    int i = 0;
    TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;

    RFIntHostApp.pF_GetGatewayUID_PANID_ADDRMODE(&gatewayuidvalid, gatewayuid, &gatewaypanid, &gatewayaddrmode, (UINT8 *)GATEWAY_DEFAULT_UID, GATEWAY_DEFAULT_PANID, GATEWAY_ADDR_MODE);
    for (i = 0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)
    {
        if ( pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY )
        {
            if( memcmp((UINT8*)pdst->IDInfo.uid, gatewayuid, UID_MAX_SIZE) == 0 )
            {                
#ifdef NEW_ITEM_OF_CTRL_CALLBK
                RF_RM_SendAnalogJoinsToApp(RF_ROAMING_GW_CONNECTED_INDEX, i+1);
#endif
                break;
            }
        }
    }

}
/**********************************************************************************************/
/*               End of RF roaming section                                                    */
/**********************************************************************************************/
